package adinsertion

import (
	"encoding/base64"
	"errors"
	"fmt"
	"sync/atomic"
	"testing"

	"github.com/golang/protobuf/proto"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"code.justin.tv/common/golibs/pubsubclient"
	"code.justin.tv/video/protocols/hlsext"
)

// Creating a new client should call Subscribe and not error
func TestNewClient(t *testing.T) {
	backend := newMockPubsubBackend()

	const testTopic = "topic"
	const testToken = "token"

	backend.SubscribeStub = func(topic, authToken string) error {
		if topic != testTopic {
			return fmt.Errorf("wrong topic: have %q, want %q", topic, testTopic)
		}
		if authToken != testToken {
			return fmt.Errorf("wrong authToken: have %q, want %q", authToken, testToken)
		}
		return nil
	}

	client, err := newPubsubClient(backend, testTopic, testToken)
	require.NoError(t, err)
	assert.NotNil(t, client)
	assert.Equal(t, 1, backend.SubscribeCount())
}

// When the Subscribe method on the backend errors, newPubsubClient
// should error.
func TestNewClientFailsToSubscribe(t *testing.T) {
	backend := newMockPubsubBackend()
	var subscribeErr = errors.New("subscribe failed")
	backend.SubscribeStub = func(_, _ string) error {
		return subscribeErr
	}

	client, err := newPubsubClient(backend, "", "")
	assert.Equal(t, subscribeErr, err)
	assert.Nil(t, client)
}

// Next() should pass through the message from the backend's
// NextMessage method
func TestClientNext(t *testing.T) {
	inputReq := &hlsext.AdBreakRequest{
		AdData: []byte("data"),
	}
	marshaledReq, err := proto.Marshal(inputReq)
	require.NoError(t, err)

	backend := newMockPubsubBackend()
	backendMsg := &pubsubclient.Message{
		Topic:   "",
		Message: base64.StdEncoding.EncodeToString(marshaledReq),
	}
	backend.NextMessageStub = func() (*pubsubclient.Message, error) {
		return backendMsg, nil
	}

	client, err := newPubsubClient(backend, "", "")
	require.NoError(t, err)
	require.NotNil(t, client)

	req, err := client.Next()
	assert.Equal(t, 1, backend.NextMessageCount())
	require.NoError(t, err)
	assert.Equal(t, inputReq, req)
}

// When the backend's NextMessage() errors, the error should bubble up
// through the client.
func TestClientNextError(t *testing.T) {
	backend := newMockPubsubBackend()
	nextMessageErr := errors.New("NextMessage failed")
	backend.NextMessageStub = func() (*pubsubclient.Message, error) {
		return nil, nextMessageErr
	}

	client, err := newPubsubClient(backend, "", "")
	require.NoError(t, err)
	require.NotNil(t, client)

	req, err := client.Next()
	assert.Nil(t, req)
	assert.Equal(t, nextMessageErr, err)
}

// Next() should return an error if it gets data that can't be
// proto unmarshaled into an AdBreakRequest.
func TestClientNextBadProtoData(t *testing.T) {
	backend := newMockPubsubBackend()
	backendMsg := &pubsubclient.Message{
		Topic:   "",
		Message: base64.StdEncoding.EncodeToString([]byte("bogus")),
	}
	backend.NextMessageStub = func() (*pubsubclient.Message, error) {
		return backendMsg, nil
	}

	client, err := newPubsubClient(backend, "", "")
	require.NoError(t, err)
	require.NotNil(t, client)

	_, err = client.Next()
	assert.Equal(t, 1, backend.NextMessageCount())
	assert.Error(t, err)
}

// Next() should return an error if it gets data that can't be
// base64-decoded.
func TestClientNextBadBase64Encoding(t *testing.T) {
	backend := newMockPubsubBackend()
	backendMsg := &pubsubclient.Message{
		Topic:   "",
		Message: string([]byte{255}),
	}
	backend.NextMessageStub = func() (*pubsubclient.Message, error) {
		return backendMsg, nil
	}

	client, err := newPubsubClient(backend, "", "")
	require.NoError(t, err)
	require.NotNil(t, client)

	_, err = client.Next()
	assert.Equal(t, 1, backend.NextMessageCount())
	assert.Error(t, err)
}

// When the backend's NextMessage() errors, if the client is closed,
// the error should be ignored.
func TestClientNextErrorWhileClosed(t *testing.T) {
	backend := newMockPubsubBackend()
	nextMessageErr := errors.New("NextMessage failed")
	backend.NextMessageStub = func() (*pubsubclient.Message, error) {
		return nil, nextMessageErr
	}

	client, err := newPubsubClient(backend, "", "")
	require.NoError(t, err)
	require.NotNil(t, client)

	err = client.Close()
	require.NoError(t, err)

	req, err := client.Next()
	assert.Nil(t, req)
	assert.Nil(t, err)
}

// Close() should pass through the backend's close error
func TestClientClose(t *testing.T) {
	backend := newMockPubsubBackend()
	closeErr := errors.New("Close failed")
	backend.CloseStub = func() error {
		return closeErr
	}

	client, err := newPubsubClient(backend, "", "")
	require.NoError(t, err)
	require.NotNil(t, client)

	err = client.Close()
	assert.Equal(t, closeErr, err)
	assert.Equal(t, 1, backend.CloseCount())
}

// Calling Close() repeatedly should just call backend.Close() once.
func TestClientCloseRepeatedly(t *testing.T) {
	backend := newMockPubsubBackend()
	closeErr := errors.New("Close failed")
	backend.CloseStub = func() error {
		return closeErr
	}

	client, err := newPubsubClient(backend, "", "")
	require.NoError(t, err)
	require.NotNil(t, client)

	err = client.Close()
	assert.Equal(t, closeErr, err)
	assert.Equal(t, 1, backend.CloseCount())

	err = client.Close()
	assert.Equal(t, nil, err)
	assert.Equal(t, 1, backend.CloseCount())

	err = client.Close()
	assert.Equal(t, nil, err)
	assert.Equal(t, 1, backend.CloseCount())
}

// mockPubsubBackend implements pubsubclient.Client
type mockPubsubBackend struct {
	SubscribeCountP   *int64
	UnsubscribeCountP *int64
	NextMessageCountP *int64
	CloseCountP       *int64

	SubscribeStub   func(topic, authToken string) error
	UnsubscribeStub func(topic string) error
	NextMessageStub func() (*pubsubclient.Message, error)
	CloseStub       func() error
}

func newMockPubsubBackend() *mockPubsubBackend {
	return &mockPubsubBackend{
		SubscribeCountP:   new(int64),
		UnsubscribeCountP: new(int64),
		NextMessageCountP: new(int64),
		CloseCountP:       new(int64),
	}
}

func (m *mockPubsubBackend) SubscribeCount() int   { return int(atomic.LoadInt64(m.SubscribeCountP)) }
func (m *mockPubsubBackend) UnsubscribeCount() int { return int(atomic.LoadInt64(m.UnsubscribeCountP)) }
func (m *mockPubsubBackend) NextMessageCount() int { return int(atomic.LoadInt64(m.NextMessageCountP)) }
func (m *mockPubsubBackend) CloseCount() int       { return int(atomic.LoadInt64(m.CloseCountP)) }

func (m *mockPubsubBackend) Subscribe(topic, authToken string) error {
	atomic.AddInt64(m.SubscribeCountP, 1)
	if m.SubscribeStub != nil {
		return m.SubscribeStub(topic, authToken)
	}
	return nil
}
func (m *mockPubsubBackend) Unsubscribe(topic string) error {
	atomic.AddInt64(m.UnsubscribeCountP, 1)
	if m.UnsubscribeStub != nil {
		return m.UnsubscribeStub(topic)
	}
	return nil
}
func (m *mockPubsubBackend) NextMessage() (*pubsubclient.Message, error) {
	atomic.AddInt64(m.NextMessageCountP, 1)
	if m.NextMessageStub != nil {
		return m.NextMessageStub()
	}
	return nil, nil
}
func (m *mockPubsubBackend) Close() error {
	atomic.AddInt64(m.CloseCountP, 1)
	if m.CloseStub != nil {
		return m.CloseStub()
	}
	return nil
}

var _ pubsubclient.Client = new(mockPubsubBackend)
