/*
Package avdata holds the container structs for libavtwitch's output data
*/
package avdata

// Codec stores the codec information for a segmented video/audio stream
type Codec struct {
	Label          string `json:"label"`
	FilenamePrefix string `json:"filename_prefix"`
	Segmentable    bool   `json:"segmentable"`

	// Audio Codec only
	AudioCodec string  `json:"audio_codec,omitempty"`
	Channels   int32   `json:"channels,omitempty"`
	SampleRate float64 `json:"sample_rate,omitempty"`

	// Video Codec only
	VideoCodec    string  `json:"video_codec,omitempty"`
	Width         float32 `json:"width,omitempty"`
	Height        float32 `json:"height,omitempty"`
	Level         int     `json:"level,omitempty"`
	Compatibility int32   `json:"compatibility,omitempty"`
	Profile       string  `json:"profile,omitempty"`
	Fps           string  `json:"fps,omitempty"`
	Vfr           string  `json:"vfr,omitempty"`
}

// Thumbnail metadata for a generated thumb by libavtwitch
type Thumbnail struct {
	Label     string `json:"label"`
	Path      string `json:"thumb_name"` //Absolute file path
	Size      int64  `json:"size"`
	Timestamp int32  `json:"timestamp"`
}

// TwitchTranscoderLogData log messages from libav
type TwitchTranscoderLogData struct {
	Type    string `json:"type"`
	Topic   string `json:"topic"`
	Message string `json:"message"`
}

// TwitchTranscoderLog general message log from TwitchTranscoder
type TwitchTranscoderLog struct {
	TTLData TwitchTranscoderLogData `json:"TWITCHTRANSCODERLOG"`
}

// StreamlogData from libavtwitch
type StreamlogData struct {
	SourceIDR int64 `json:"source_idr_interval"`
}

// StreamLog from libavtwitch
type StreamLog struct {
	SLData StreamlogData `json:"STREAMLOG"`
}

// VideoQualityMetrics collects metrics like PSNR for a segment
type VideoQualityMetrics struct {
	PsnrU    float32 `json:"PSNR_U"`
	PsnrV    float32 `json:"PSNR_V"`
	PsnrY    float32 `json:"PSNR_Y"`
	PsnrAvg  float32 `json:"PSNR_avg"`
	Ssim     float32 `json:"SSIM"`
	FrameCnt int32   `json:"frame_cnt"`
}

// VideoEncoderMetrics collects metrics like profile for a segment
type VideoEncoderMetrics struct {
	Profile           string  `json:"profile"`
	Preset            string  `json:"preset"`
	Bitrate           int32   `json:"bitrate"`
	FrameCnt          int32   `json:"frame_cnt"`
	FrameOutOfSyncCnt int32   `json:"frame_out_of_sync_cnt"`
	Fps               float32 `json:"fps"`
}

// AvVersions holds all video libraries version information
type AvVersions struct {
	AvcodecVersion    string `json:"avcodec_version"`
	AvformatVersion   string `json:"avformat_version"`
	AvtwitchVersion   string `json:"avtwitch_version"`
	AvutilVersion     string `json:"avutil_version"`
	FdkaacencVersion  string `json:"fdkaacenc_version"`
	LibrtmpVersion    string `json:"librtmp_version"`
	Libx264Version    string `json:"libx264_version"`
	SwresampleVersion string `json:"swresample_version"`
	SwscaleVersion    string `json:"swscale_version"`
}

// Segment represents a video or audio segment produced by libavtwitch
type Segment struct {
	Label              string              `json:"label"`
	SegmentNumber      int64               `json:"segment_number"`
	SegmentSize        int64               `json:"segment_size"`
	SegmentName        string              `json:"segment_name"`
	FutureSegments     []string            `json:"future_segments"`
	Overhead           string              `json:"path"`
	OverheadPercentage float32             `json:"segment_overhead_percent"`
	Latency            float32             `json:"latency"`
	Duration           float64             `json:"duration"`
	Bandwidth          int32               `json:"bandwidth"`
	FrameCount         int32               `json:"frame_count"`
	IdrCount           int32               `json:"idr_count"`
	Timestamp          int32               `json:"timestamp"`
	MaxIdrDelta        float64             `json:"max_idr_delta"`
	RandomAccessPoint  bool                `json:"random_access_point"`
	VqMetrics          VideoQualityMetrics `json:"VideoQualityMetrics"`
	VeMetrics          VideoEncoderMetrics `json:"VideoEncoderMetrics"`
}

// ProbeVideo response from libavtwitch
type ProbeVideo struct {
	Bitrate int64   `json:"bitrate"`
	Codec   string  `json:"codec"`
	Fps     float32 `json:"fps"`
	Profile string  `json:"profile"`
	Height  float32 `json:"height"`
	Width   float32 `json:"width"`
}

// ProbeAudio response from libavtwitch
type ProbeAudio struct {
	Bitrate    int64  `json:"bitrate"`
	Codec      string `json:"codec"`
	Channels   int    `json:"channels"`
	SampleRate int32  `json:"sample_rate"`
}

// ProbeAV response from libavtwitch
type ProbeAV struct {
	Video ProbeVideo
	Audio ProbeAudio
	Error string `json:"error"`
}

// ProbeResult from libavtwitch
type ProbeResult struct {
	ProbeAV ProbeAV `json:"twitch_probe_result"`
}

// DebugMarkers stores fields to find and parse in libavtwitch's output
// This is a map of key values to iterate over any lookup fields
// In a following version this messages should be standarized, possibly json
var DebugMarkers = map[string]string{
	"TimeScale":          "---time_scale: ",
	"UnitsInTick":        "---num_units_in_tick: ",
	"VideoFormat":        "---video_format: ",
	"VideoFullRangeFlag": "---video_full_range_flag: ",
	"FixedFramerateFlag": "---fixed_frame_rate_flag: ",
	"ColourPrimaries":    "---colour_primaries: ",
	"Resolution":         "Res:",
	"Width":              "width=",
	"Height":             "height=",
	"Bitrate":            "bitrate=",
	"Preset":             "preset=",
	"Level":              "level=",
	"Codec":              "codec=",
}
