/*
Package config to deserialize json based runtime parameters

*/
package config

import (
	"encoding/json"
	"io/ioutil"
	"log"
	"os"
)

// Configuratin struct.
// All  elements of configuration should be declared in this block

// ArchivesConfig settings related to archiving video chunks
type ArchivesConfig struct {
	Path string `json:"path"`
}

// StatsdConfig settings
type StatsdConfig struct {
	Enabled     bool   `json:"enabled"`
	Host        string `json:"host"`
	Port        string `json:"port"`
	ServiceName string `json:"service_name"`
}

// UsherConfig settings
type UsherConfig struct {
	Host string `json:"host"`
}

// DigestionConfig settings
type DigestionConfig struct {
	Host string `json:"host"`
}

// twitchtranscoder encoder_type options:
//   - x264
//   - quicksync

// TwitchTranscoderConfig transcoding settings for TwitchTranscoder
type TwitchTranscoderConfig struct {
	PathStable         string `json:"path_stable"`
	PathBeta           string `json:"path_beta"`
	TranscodeProfile   string `json:"transcode_profile"`
	ChunkPrefix        string `json:"chunk_prefix"`
	ThumbDelay         int32  `json:"thumb_delay"`
	ThumbInterval      int32  `json:"thumb_interval"`
	SegmentMinDuration int    `json:"segment_min_duration"`
	SegmentMaxDuration int    `json:"segment_max_duration"`
	EncoderType        string `json:"encoder_type"`
}

//ElasticSearchConfig holds data server settings
type ElasticSearchConfig struct {
	Host               string  `json:"host"`
	Port               int32   `json:"port"`
	Enable             bool    `json:"enable"`
	SampleRate         float64 `json:"sample_rate"`
	IndexRetentionDays int32   `json:"index_retention_days"`
}

// TelegraphConfig holds telegraph settings
type TelegraphConfig struct {
	Host              string `json:"host"`
	Port              int32  `json:"port"`
	ConnectionTimeout int32  `json:"connection_timeout"`
	ConnectionRetry   int32  `json:"connection_retry"`
}

// VODConfig holds VOD pusher settings
type VODConfig struct {
	QueueSize             int     `json:"queue_size"`
	NotifyQueueSize       int     `json:"notify_queue_size"`
	NumThumbnails         int     `json:"num_thumbnails"`
	TargetObjectDur       float64 `json:"target_object_dur"` //seconds
	VinylHost             string  `json:"vinyl_host"`
	S3Bucket              string  `json:"s3_bucket"`
	Proxy                 string  `json:"proxy"`
	MinVodDurationSeconds int     `json:"min_vod_duration"`
}

// SegmentDuration defines duration min max ranges
type SegmentDuration struct {
	Min int `json:"min"` //milliseconds
	Max int `json:"max"` //milliseconds
}

// SegmentConfig holds segment duration settings
// 'TargetDuration' is a key to DurationRanges
type SegmentConfig struct {
	TargetDuration string                     `json:"target_duration,omitempty"` //seconds
	DurationRanges map[string]SegmentDuration `json:"duration_ranges"`
}

//TenfootClientSettings configures the Tenfoot eventbus client.
type TenfootClientSettings struct {
	Enabled         bool   `json:"enabled"`
	SockBasePath    string `json:"sockBasePath"`
	SendTimeout     string `json:"sendTimeout"`
	ShutdownTimeout string `json:"shutdownTimeout"`
	LogLevel        string `json:"logLevel"`
}

// AdInsertionConfig describes how to retrieve the "insert an ad"
// messages.
type AdInsertionConfig struct {
	PubsubURL string `json:"pubsub_url"`
	Proxy     string `json:"proxy"`
	Topic     string `json:"topic"`
}

// StreamlogConfig Settings
type StreamlogConfig struct {
	FramedropThreshold float64 `json:"framedrop_threshold"`
	Endpoint           string  `json:"endpoint"`
	ServiceTag         string  `json:"service_tag"`
	Proxy              string  `json:"proxy"`
	QueueSize          int     `json:"queue_size"`
}

// Ingest notifications service settings
type NotificationsConfig struct {
	StreamEventTopicARN string `json:"stream_event_topic_arn"`
}

type OriginConfig struct {
	OriginEndpoint string `json:"origin_endpoint"`
}

// Values is an aggregate Struct. This contains all settings blocks for json unmarshalling
type Values struct {
	Archives         ArchivesConfig         `json:"archives"`
	Statsd           StatsdConfig           `json:"statsd"`
	Usher            UsherConfig            `json:"usher"`
	TwitchTranscoder TwitchTranscoderConfig `json:"twitchtranscoder"`
	Telegraph        TelegraphConfig        `json:"telegraph"`
	ElasticSearch    ElasticSearchConfig    `json:"elasticsearch"`
	VOD              VODConfig              `json:"vod"`
	Segment          SegmentConfig          `json:"segment"`
	Ads              AdInsertionConfig      `json:"ads"`
	Tenfoot          TenfootClientSettings  `json:"tenfoot"`
	Streamlog        StreamlogConfig        `json:"streamlog"`
	Noisy            NoisyConfig            `json:"noisy"`
	Notifications    NotificationsConfig    `json:"notifications"`
	Origin           OriginConfig           `json:"origin"`
}

// NoisyConfig specifies whether we should output to rollbar
type NoisyConfig struct {
	RollbarToken string `json:"rollbar_token"`
}

// Load the configurataion from a configuration file.
// Returns an error if the config has already been initialized
// TODO: only let config load once in a way that doesn't break parallelized tests
func Load(configurationFile string, config interface{}) error {

	if _, err := os.Stat(configurationFile); err != nil {
		log.Println(err)
		return err
	}

	file, err := ioutil.ReadFile(configurationFile)

	if err != nil {
		log.Println("Error loading config: ", err)
		return err
	}

	if err = json.Unmarshal(file, config); err != nil {
		log.Println("Failed to parse config file", configurationFile)
		return err
	}

	return nil
}
