package digestion

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net/http"
)

// Settings contains the digestion configuration
type Settings struct {
	BaseURL string
}

// Post to digestion's api
func (cfg *Settings) Post(path string, data interface{}) ([]byte, error) {

	var bodyReader io.Reader

	if data != nil {
		body, err := json.Marshal(data)
		if err != nil {
			return nil, err
		}
		bodyReader = bytes.NewBuffer(body)
	}

	client := &http.Client{}

	req, err := http.NewRequest("POST", path, bodyReader)
	if err != nil {
		return nil, err
	}

	req.Header.Add("Content-Type", `application/json`)

	response, err := client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = response.Body.Close()
	}()

	if response.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("[Digestion API] Failed to perform digestion request.\n Status[%d].\n Path [%s].\n Response [%v] \n", response.StatusCode, path, response)
	}

	contents, err := ioutil.ReadAll(response.Body)
	if err != nil {
		log.Printf("[Digestion API] Failed to get body of response %s\n", err)
		return nil, err
	}

	return contents, nil
}

// StreamInfo will request the rtmp url from Digestion
func (cfg *Settings) StreamInfo(channel string) (*Stream, error) {
	var info Stream

	request := StreamRequest{
		Channel: channel,
		Format:  "live",
	}

	data, err := cfg.Post(fmt.Sprintf("%s/v1/DigestionV1/GetStream", cfg.BaseURL), request)
	if err != nil {
		log.Printf("Failed to retrieve stream information from digestion: %s, %+v \n", err, info)
		return nil, err
	}

	if err = json.Unmarshal(data, &info); err != nil {
		log.Printf("Failed to unmarshall digestion response: %s, %s \n", err, data)
		return nil, err
	}

	return &info, nil
}
