/*
Package elasticsearch will output metatada via elasticsearch api to log structs as metadata
*/
package elasticsearch

import (
	"bytes"
	"encoding/json"
	"fmt"
	"log"
	"math/rand"
	"net/http"
	"sync"
	"time"
)

const (
	index   = "gotranscoder-%d-%v-%d"
	httpURL = `http://%s:%d/%s/%s`
)

const writeBufferSize = 100

var mutex = &sync.Mutex{}

// buffer the http writes
var writeBuffer = make(chan Log, writeBufferSize)

// Settings contains ES server configuration
type Settings struct {
	Host               string
	Port               int32
	SampleRate         float64
	IndexRetentionDays int32
}

// Log holds a message
type Log struct {
	MessageType string
	Channel     string
	ChannelID   uint64
	Data        interface{}
}

// Initialize the ElasticSearch writer
func (cfg *Settings) Initialize() {
	go cfg.ReadLogs()
	cfg.ExpireIndex()
}

// IndexKey generates the insert index name
func IndexKey(indexTime time.Time) string {
	year, month, day := indexTime.Date()
	return fmt.Sprintf(index, year, int(month), day)
}

// ExpireIndex removes old elastic search indexes
// Nothing will happen if this is called more than once, but ideally this lives
// in the future cleanup worker.
func (cfg *Settings) ExpireIndex() {
	lastWeek := time.Now().AddDate(0, 0, -int(cfg.IndexRetentionDays))
	url := fmt.Sprintf(httpURL, cfg.Host, cfg.Port, IndexKey(lastWeek), "")
	log.Printf("Expiring Elastic Search index: %s\n", url)

	request, err := http.NewRequest("DELETE", url, nil)
	if err != nil {
		log.Printf("failed to perform elasticsearch index delete, %s\n", err)
	}

	response, err := http.DefaultClient.Do(request)
	if err != nil {
		log.Printf("failed to perform elasticsearch index delete, %s\n", err)
		return
	}
	_ = response.Body.Close()
}

// ShouldSample returns randomly if a write needs to be sampled
func (cfg *Settings) ShouldSample() bool {
	return (rand.Float64() < cfg.SampleRate)
}

// ReadLogs loops over the buffered channel and logs into Elastic search when a message is received
func (cfg *Settings) ReadLogs() {
	for {
		select {
		case entry := <-writeBuffer:
			err := cfg.WriteHTTP(&entry)
			if err != nil {
				log.Printf(" Error pushing elasticsearch stat via HTTP: %s", err)
			}
		}
		time.Sleep(50 * time.Millisecond)
	}
}

// WriteLog a log line into elasticsearch only if there is space in the buffer.
// Ignore message if buffer is full
func (cfg *Settings) WriteLog(logLine *Log, sample bool) {
	mutex.Lock()
	defer mutex.Unlock()

	if len(writeBuffer) < writeBufferSize {
		if sample == false || (sample && cfg.ShouldSample()) {
			writeBuffer <- *logLine
		}
	} else {
		log.Println("ElasticSearch write buffer full: dropping message")
	}
}

// WriteHTTP pushes a json string into elasticsearch
func (cfg *Settings) WriteHTTP(logLine *Log) error {
	esData := make(map[string]interface{})

	url := fmt.Sprintf(httpURL, cfg.Host, cfg.Port, IndexKey(time.Now()), logLine.MessageType)

	esData["@timestamp"] = time.Now()
	esData["@channel"] = logLine.Channel
	esData["@channel_id"] = logLine.ChannelID
	esData[logLine.MessageType] = logLine.Data

	body, err := json.Marshal(esData)
	if err != nil {
		log.Printf("failed generate elasticsearch request, %s\n", err)
		return err
	}

	request, err := http.NewRequest("POST", url, bytes.NewBuffer(body))

	if err != nil {
		log.Printf("failed to POST elasticsearch request, %s\n", err)
		return err
	}
	request.Header.Set("Content-Type", "application/json")

	client := &http.Client{}
	response, err := client.Do(request)
	if err != nil {
		log.Printf("failed to perform elasticsearch request, %s\n", err)
		return err
	}
	_ = response.Body.Close()
	return nil
}
