package elasticsearch

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"strconv"
	"strings"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
)

// MockElasticSearch returns a mock http server and a settings object pointing to such server
func MockElasticSearch() (*httptest.Server, Settings) {

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(200)
		w.Header().Set("Content-Type", "application/json")
		fmt.Fprintln(w, `[{"ALL GOOD"}]`)
	}))

	iport, _ := strconv.ParseInt(strings.Split(server.URL, ":")[2], 0, 32)

	local := Settings{
		Host:       "127.0.0.1",
		Port:       int32(iport),
		SampleRate: 1.0,
	}

	return server, local
}

func TestWrite(t *testing.T) {
	assert := assert.New(t)
	server, settings := MockElasticSearch()
	defer server.Close()

	testLog := Log{
		MessageType: "test",
		Channel:     "test",
		Data:        "test",
	}

	err := settings.WriteHTTP(&testLog)
	assert.NoError(err)

}

func TestSampleRate(t *testing.T) {
	assert := assert.New(t)
	server, settings := MockElasticSearch()
	defer server.Close()

	settings.SampleRate = 0
	assert.False(settings.ShouldSample())

	settings.SampleRate = 1
	assert.True(settings.ShouldSample())
}

func TestExpiration(t *testing.T) {
	assert := assert.New(t)
	server, settings := MockElasticSearch()
	defer server.Close()

	settings.SampleRate = 0
	assert.NotPanics(func() { settings.ExpireIndex() })

}

func TestReaderWriter(t *testing.T) {
	assert := assert.New(t)
	server, settings := MockElasticSearch()
	settings.SampleRate = 0
	defer server.Close()

	log := Log{
		MessageType: "test",
		Channel:     "testchannel",
		Data:        "data",
	}

	settings.WriteLog(&log, false)
	settings.WriteLog(&log, false)
	assert.Equal(len(writeBuffer), 2)

	go settings.ReadLogs()
	time.Sleep(500 * time.Millisecond)
	assert.Equal(len(writeBuffer), 0)
}

func TestFailWrite(t *testing.T) {
	assert := assert.New(t)
	server, settings := MockElasticSearch()
	server.Close()

	testLog := Log{
		MessageType: "test",
		Channel:     "test",
		Data:        "test",
	}

	err := settings.WriteHTTP(&testLog)
	assert.Error(err)
}
