package m3u8

import (
	"fmt"
	"testing"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"github.com/stretchr/testify/assert"
)

func generateTestChunks(prefix string, count int, chunkDuration float64) []Chunk {
	var chunks = make([]Chunk, count)
	for index := 0; index < count; index++ {
		chunks[index] = Chunk{
			SequenceNumber: int64(index),
			URL:            fmt.Sprintf("index-%d.ts", index),
			Duration:       chunkDuration,
		}
	}
	return chunks
}

func generateTestSegments(chunks *[]Chunk) []avdata.Segment {
	segments := make([]avdata.Segment, 0, 50)
	for _, val := range *chunks {
		segment := avdata.Segment{
			Duration: val.Duration,
		}
		segments = append(segments, segment)
	}
	return segments
}

func assertHeaderFormat(t *testing.T, output string) error {
	assert := assert.New(t)

	assert.NotNil(output)
	assert.Contains(output, "#EXTM3U")
	assert.Contains(output, "#EXT-X-VERSION")
	assert.Contains(output, "#EXT-X-TARGETDURATION")
	assert.Contains(output, "#ID3-EQUIV-TDTG")
	assert.Contains(output, "#EXT-X-MEDIA-SEQUENCE")

	return nil
}

func TestPlaylistHeader(t *testing.T) {
	assert := assert.New(t)

	// starting playlist
	var testPlaylist = Playlist{
		StreamOffset:   0,
		StreamDuration: 0,
		TargetDuration: 0,
		WindowSize:     5,
		MediaSequence:  0,
	}

	// Empty playlist
	output := string(testPlaylist.playlistHeader())
	assert.NoError(assertHeaderFormat(t, output))

	// With chunks
	testPlaylist.Chunks = generateTestChunks("index", 10, 4100)
	output = string(testPlaylist.playlistHeader())
	assert.NoError(assertHeaderFormat(t, output))

	// Set media sequence
	testPlaylist.MediaSequence = 5
	output = string(testPlaylist.playlistHeader())
	assert.NoError(assertHeaderFormat(t, output))
	assert.Contains(output, "#EXT-X-MEDIA-SEQUENCE:5")

	// Target duration rounding
	testPlaylist.TargetDuration = 5
	output = string(testPlaylist.playlistHeader())
	assert.NoError(assertHeaderFormat(t, output))
	assert.Contains(output, "#EXT-X-TARGETDURATION:5")

	// Event type playlists
	testPlaylist.PlaylistType = PlaylistTypeLive
	output = string(testPlaylist.playlistHeader())
	assert.NoError(assertHeaderFormat(t, output))
	assert.NotContains(output, "#EXT-X-PLAYLIST-TYPE")
	testPlaylist.PlaylistType = PlaylistTypeVod
	output = string(testPlaylist.playlistHeader())
	assert.NoError(assertHeaderFormat(t, output))
	assert.Contains(output, "#EXT-X-PLAYLIST-TYPE:VOD")
	testPlaylist.PlaylistType = PlaylistTypeEvent
	output = string(testPlaylist.playlistHeader())
	assert.NoError(assertHeaderFormat(t, output))
	assert.Contains(output, "#EXT-X-PLAYLIST-TYPE:EVENT")

}

func assertTwitchHeaderFormat(t *testing.T, output string) error {
	assert := assert.New(t)

	assert.NotNil(output)
	assert.Contains(output, "#EXT-X-TWITCH-TOTAL-SECS")
	assert.Contains(output, "#EXT-X-TWITCH-ELAPSED-SEC")

	return nil
}

func TestTwitchPlaylistHeaders(t *testing.T) {
	assert := assert.New(t)

	// starting playlist
	var testPlaylist = Playlist{
		StreamOffset:   0,
		StreamDuration: 0,
		TargetDuration: 0,
		WindowSize:     5,
		MediaSequence:  0,
	}

	// Empty playlist
	output := string(testPlaylist.twitchPlaylistHeader())
	assert.NoError(assertTwitchHeaderFormat(t, output))
	assert.Contains(output, "ELAPSED-SECS:0.000")
	assert.Contains(output, "TOTAL-SECS:0.000")

	// test for #EXT-X-TWITCH-ELAPSED-SYSTEM-SECS
	testPlaylist.Chunks = generateTestChunks("index", 10, 1000)
	testPlaylist.StreamDuration = 100

	// test for #EXT-X-TWITCH-ELAPSED-SECS
	var testPlaylist2 = Playlist{
		StreamOffset:   0,
		StreamDuration: 0,
		TargetDuration: 0,
		WindowSize:     5,
		MediaSequence:  0,
	}
	testPlaylist2.Chunks = generateTestChunks("index", 100, 1000)
	segments := generateTestSegments(&testPlaylist2.Chunks)
	streamOffset := &StreamOffset{}
	Elapsed(streamOffset, &segments, 90)
	testPlaylist.StreamOffset = streamOffset.TimeElapsed

	output = string(testPlaylist.twitchPlaylistHeader())
	assert.NoError(assertTwitchHeaderFormat(t, output))
	assert.Contains(output, "ELAPSED-SECS:90.000")
	assert.Contains(output, "TOTAL-SECS:100.000")

}

func TestV3Playlist(t *testing.T) {
	assert := assert.New(t)

	// starting playlist
	var testPlaylist = Playlist{
		StreamOffset:   0,
		StreamDuration: 0,
		TargetDuration: 0,
		WindowSize:     5,
		MediaSequence:  0,
		IsFinal:        false,
	}

	// Empty playlist
	output := string(testPlaylist.GenerateV3())

	assert.NoError(assertTwitchHeaderFormat(t, output))
	assert.NoError(assertHeaderFormat(t, output))
	assert.NotContains(output, "END")
	assert.NotContains(output, "DISCONTINUITY")

	// 5 initial chunks
	testPlaylist.PlaylistType = PlaylistTypeLive
	testPlaylist.Chunks = generateTestChunks("index", 5, 4100)
	testPlaylist.TargetDuration = 5
	testPlaylist.StreamDuration = 21

	output = string(testPlaylist.GenerateV3())
	assert.Contains(output, "#EXTINF:4.100,")
	assert.Contains(output, "#EXTINF:4.100,")
	assert.Contains(output, "index-0.ts")
	assert.Contains(output, "index-4.ts")
	assert.NotContains(output, "index-5.ts")
	assert.NotContains(output, "END")
	assert.NotContains(output, "DISCONTINUITY")

	// 35 chunks
	testPlaylist.PlaylistType = PlaylistTypeLive
	testPlaylist.Chunks = generateTestChunks("index", 40, 4100)
	testPlaylist.MediaSequence = 35

	output = string(testPlaylist.GenerateV3())
	assert.Contains(output, "#EXTINF:4.100,")
	assert.Contains(output, "#EXTINF:4.100,")
	assert.Contains(output, "index-30.ts")
	assert.Contains(output, "index-34.ts")
	assert.NotContains(output, "index-41.ts")
	assert.NotContains(output, "END")
	assert.NotContains(output, "DISCONTINUITY")

	// discontinuity
	testPlaylist.Chunks[5].URL = DiscontinuityURL
	output = string(testPlaylist.GenerateV3())

	assert.NotContains(output, "discontinuity")
	assert.Contains(output, "#EXT-X-DISCONTINUITY")
	assert.Contains(output, "#EXT-X-TWITCH-DISCONTINUITY")
	assert.NotContains(output, "END")

	// check end of playlist
	testPlaylist.IsFinal = true
	output = string(testPlaylist.GenerateV3())
	assert.Contains(output, "#EXT-X-ENDLIST")

}
