package notify

import (
	"fmt"
	"log"
	"time"

	"code.justin.tv/video/gotranscoder/pkg/statsd"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sns/snsiface"
)

//go:generate counterfeiter -o fakes/fake_sns.go "../../vendor/github.com/aws/aws-sdk-go/service/sns/snsiface" SNSAPI

const (
	defaultAWSRegion = "us-west-2"
	statsdSNSRetry   = "notify.sns%s.retry"
	statsdSNSRespFmt = "notify.sns%s.%d"
	statsdTimingFmt  = "notify.sns%s.%s.%s"
)

// SNS is a wrapper for the aws sdk's sns client
type SNS interface {
	Publish(msg, subject, topic string) error
}

type snsImpl struct {
	svc         snsiface.SNSAPI
	backendName string
}

// NewSNS creates a wrapper based on the passed in SNS client
func NewSNS(snsapi snsiface.SNSAPI, backendName string) SNS {
	sess, _ := session.NewSession()
	sess.Handlers.Retry.PushBackNamed(request.NamedHandler{
		Name: "statsd",
		Fn: func(r *request.Request) {
			statsd.Inc(fmt.Sprintf(statsdSNSRetry, backendName), 1, 1.0)
			statusCode := fmt.Sprintf(statsdSNSRespFmt, backendName, r.HTTPResponse.StatusCode)
			statsd.Inc(statusCode, 1, 1.0) //notify.sns.$STATUSCODE
			if awsErr, ok := r.Error.(awserr.Error); ok {
				log.Println("[NOTIFY][AWSERROR]:", awsErr.Code(), awsErr.Message(), awsErr.OrigErr())
			}
		},
	})

	sess.Handlers.ValidateResponse.PushBackNamed(request.NamedHandler{
		Name: "statsd",
		Fn: func(r *request.Request) {
			statusCode := fmt.Sprintf(statsdSNSRespFmt, backendName, r.HTTPResponse.StatusCode)
			statsd.Inc(statusCode, 1, 1.0) //notify.sns.$STATUSCODE
		},
	})

	var svc snsiface.SNSAPI
	if snsapi != nil {
		svc = snsapi
	} else {
		svc = sns.New(sess, aws.NewConfig().WithRegion(defaultAWSRegion).WithMaxRetries(3))
	}

	return &snsImpl{
		svc:         svc,
		backendName: backendName,
	}
}

// Publish emits an sns notification based on the pre-configured parameters
func (s *snsImpl) Publish(msg, subject, topic string) error {
	params := &sns.PublishInput{
		Message:  aws.String(msg),
		Subject:  aws.String(subject),
		TopicArn: aws.String(topic),
	}
	// track timings of sns notifications
	start := time.Now()
	_, err := s.svc.Publish(params)
	statsd.Timing(fmt.Sprintf(statsdTimingFmt, s.backendName, subject, topic), time.Since(start), 1.0)
	return err
}
