package origin_test

import (
	"fmt"
	"net/http/httptest"
	"strings"

	"golang.org/x/net/context"

	. "code.justin.tv/video/gotranscoder/pkg/origin"
	"code.justin.tv/video/origin/rpc/originctl"
	"code.justin.tv/video/origin/rpc/originctl/originctlfakes"
	. "github.com/onsi/ginkgo"
	"github.com/onsi/ginkgo/reporters"
	. "github.com/onsi/gomega"

	"testing"
)

var _ = Describe("Origin", func() {
	var (
		control       *originctlfakes.FakeOriginControlV1
		server        *httptest.Server
		client        Origin
		ctx           context.Context
		goodChannelId = "asdf-zxcv"
		goodSessionId = "12345"
		goodSession   = &originctl.Session{Id: goodSessionId}
	)
	BeforeEach(func() {
		control = &originctlfakes.FakeOriginControlV1{}
		server = httptest.NewServer(originctl.NewOriginControlV1Server(control, nil, nil))
		client = New(Config{Endpoint: server.URL})
		ctx = context.Background()
	})

	Context("when Origin is working", func() {
		BeforeEach(func() {
			control.StartSessionReturns(&originctl.StartSessionResponse{
				Session: goodSession,
			}, nil)
			control.TerminateSessionReturns(&originctl.TerminateSessionResponse{}, nil)
		})
		It("runs the session correctly", func() {
			By("Starting the session correctly")
			session, exitFunction, err := client.StartSession(ctx, goodChannelId, []string{"chunk-only"})
			Expect(err).To(BeNil())
			Expect(session).To(Equal(goodSession))
			Expect(exitFunction).NotTo(BeNil())
			Expect(control.TerminateSessionCallCount()).To(Equal(0))
			By("terminating the session correctly")
			exitFunction()
			Expect(control.TerminateSessionCallCount()).To(Equal(1))
			_, terminatedSession := control.TerminateSessionArgsForCall(0)
			Expect(terminatedSession.Id).To(Equal(goodSession.Id))
		})
	})

	Context("when Origin is not working", func() {
		var originalError = fmt.Errorf("an error occurred")
		BeforeEach(func() {
			control.StartSessionReturns(nil, originalError)
		})
		It("passes back the failure", func() {
			session, exitFunction, err := client.StartSession(ctx, goodChannelId, []string{"chunk-only"})
			Expect(err).NotTo(BeNil())
			Expect(strings.Contains(err.Error(), originalError.Error())).To(BeTrue())
			Expect(exitFunction).To(BeNil())
			Expect(session).To(BeNil())
			Expect(control.TerminateSessionCallCount()).To(Equal(0))
		})
	})
})

func TestOrigin(t *testing.T) {
	RegisterFailHandler(Fail)
	junitReporter := reporters.NewJUnitReporter("../../gotranscoder_test_result.xml")
	RunSpecsWithCustomReporters(t, "Origin Suite", []Reporter{junitReporter})
}
