// Package statsd to write into graphite
package statsd

import (
	"fmt"
	"log"
	"net"
	"strings"
	"time"

	"github.com/cactus/go-statsd-client/statsd"
)

var conn statsd.Statter
var cfg Settings

// Settings for StatsD
type Settings struct {
	Env         string
	Enabled     bool
	Host        string
	Port        string
	ServiceName string
	LVS         bool
	ChannelName string
}

// Set mock conn for testing
func SetMockConnection(c statsd.Statter) {
	conn = c
}

// Connect to the statsd server
func (config *Settings) Connect() {
	cfg = *config
	if config.Enabled {
		conn = mustDial()
	}
}

func mustDial() statsd.Statter {
	statsdPrefix := cfg.ServiceName
	if cfg.Env != "" {
		statsdPrefix = statsdPrefix + "." + cfg.Env
	}
	s, err := statsd.NewBufferedClient(net.JoinHostPort(cfg.Host, cfg.Port), statsdPrefix, 0, 0)
	if err != nil {
		log.Printf("Failed to connect to StatsD. Disabling Stats - %s", err)
	}

	return s
}

// Inc increments a counter by a value and rate
func Inc(stat string, value int64, rate float32) {
	if cfg.Enabled {
		var err error
		if !cfg.LVS {
			err = conn.Inc(stat, value, rate)
		} else {
			metric := fmt.Sprintf("%s.%s.%s", "lvs", stat, strings.TrimPrefix(cfg.ChannelName, "lvs."))
			err = conn.Inc(metric, value, rate)
		}
		if err != nil {
			log.Printf("Error incrementing stat: %s - %s", stat, err)
		}
	}
}

// Gauge is used to measure a value
func Gauge(stat string, value int64, rate float32) {
	if cfg.Enabled {
		var err error
		if !cfg.LVS {
			err = conn.Gauge(stat, value, rate)
		} else {
			metric := fmt.Sprintf("%s.%s.%s", "lvs", stat, strings.TrimPrefix(cfg.ChannelName, "lvs."))
			err = conn.Gauge(metric, value, rate)
		}
		if err != nil {
			log.Printf("Error incrementing gauge: %s - %s", stat, err)
		}
	}
}

// GaugeDelta will measure the delta between values
func GaugeDelta(stat string, value int64, rate float32) {
	if cfg.Enabled {
		var err error
		if !cfg.LVS {
			err = conn.GaugeDelta(stat, value, rate)
		} else {
			metric := fmt.Sprintf("%s.%s.%s", "lvs", stat, strings.TrimPrefix(cfg.ChannelName, "lvs."))
			err = conn.GaugeDelta(metric, value, rate)
		}
		if err != nil {
			log.Printf("Error on gauge delta increment: %s - %s", stat, err)
		}
	}
}

// Timing will track timing of operations
func Timing(stat string, value time.Duration, rate float32) {
	if cfg.Enabled {
		var err error
		if !cfg.LVS {
			err = conn.Timing(stat, int64(value/time.Millisecond), rate)
		} else {
			metric := fmt.Sprintf("%s.%s.%s", "lvs", stat, strings.TrimPrefix(cfg.ChannelName, "lvs."))
			err = conn.Timing(metric, int64(value/time.Millisecond), rate)
		}
		if err != nil {
			log.Printf("Error incrementing timing stat: %s - %s", stat, err)
		}
	}
}

// CustomTiming passes through the integer time given directly
func CustomTiming(stat string, value int64, rate float32) {
	if cfg.Enabled {
		var err error
		if !cfg.LVS {
			err = conn.Timing(stat, value, rate)
		} else {
			metric := fmt.Sprintf("%s.%s.%s", "lvs", stat, strings.TrimPrefix(cfg.ChannelName, "lvs."))
			err = conn.Timing(metric, value, rate)
		}
		if err != nil {
			log.Printf("Error incrementing timing stat: %s - %s", stat, err)
		}
	}
}

// Raw will push raw data into statsd
func Raw(stat string, value string, rate float32) {
	if cfg.Enabled {
		var err error
		if !cfg.LVS {
			err = conn.Raw(stat, value, rate)
		} else {
			metric := fmt.Sprintf("%s.%s.%s", "lvs", stat, strings.TrimPrefix(cfg.ChannelName, "lvs."))
			err = conn.Raw(metric, value, rate)
		}
		if err != nil {
			log.Printf("Error incrementing raw stat: %s - %s", stat, err)
		}
	}
}
