package tenfoot

import (
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"code.justin.tv/video/gotranscoder/pkg/m3u8"
	"code.justin.tv/video/origin/rpc/originctl"
	"code.justin.tv/video/origin/rpc/originfoot"
	"code.justin.tv/video/origin/rpc/originfoot/originfootfakes"
	"code.justin.tv/video/protocols/hls"
	"code.justin.tv/video/protocols/hlsext"
	"code.justin.tv/video/protocols/tenfoot"

	uuid "github.com/satori/go.uuid"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"golang.org/x/net/context"
)

func TestOriginfootNoops(t *testing.T) {
	o := OriginFoot{
		base: Noop(),
	}
	assert.NoError(t, o.Wait())
	assert.NoError(t, o.PostSegment(nil))
}

var testMetadata = &TranscodeMetadata{
	StartTimestamp:        time.Now(),
	TargetSegmentDuration: 2 * time.Second,
	TranscodeProfile:      "Transcode2015Main",
	Qualities:             []string{"chunked", "audio_only"},
	Channel:               "foouser",
	UsherStreamID:         123,
	SessionID:             uuid.NewV4(),
	ResourcePath:          "foouser_123_456",
	OriginHostname:        "hls-28bca1",
}

var testPlaylist = m3u8.Playlist{
	Chunks: []m3u8.Chunk{
		{
			SequenceNumber: 1,
			Duration:       1000,
			URL:            "seg1.ts",
		},
	},
	StreamDuration: 1000.0,
	StreamOffset:   1000.0,
	TargetDuration: 3,
	IsFinal:        false,
	WindowSize:     6,
	MediaSequence:  2,
	PlaylistType:   m3u8.PlaylistTypeLive,
}

func TestOriginfoot(t *testing.T) {
	client := &originfootfakes.FakeOriginTenfootShim{}
	o := &OriginFoot{
		base:     Noop(),
		client:   client,
		session:  &originctl.Session{Id: "1"},
		metadata: testMetadata,
	}

	quality := "chunked"
	playlistPath := ""
	playlistType := "live"
	adBreaks := []*hlsext.AdBreakRequest{}
	futureSegments := []*hls.Segment{}

	require.NoError(t, o.PostPlaylist(testPlaylist, quality, playlistPath, playlistType, adBreaks, futureSegments))

	called := assert.Equal(t, 1, client.SetPlaylistCallCount(),
		"originfoot's SetPlaylist method should get called",
	)
	if called {
		_, req := client.SetPlaylistArgsForCall(0)
		require.NotNil(t, req)
		assert.Equal(t, o.session.Id, req.Session.Id)
		assert.Equal(t, tenfoot.NewPlaylist_M3U8_LIVE, req.Playlist.Type)
		assert.EqualValues(t, testPlaylist.WindowSize, req.Playlist.WindowSize)
		hasSeg := assert.Len(t, req.Playlist.Playlist.Playlist.Segments, 1)
		if hasSeg {
			seg := req.Playlist.Playlist.Playlist.Segments[0]
			assert.Equal(t, testPlaylist.Chunks[0].URL, seg.Uri)
			assert.EqualValues(t, testPlaylist.Chunks[0].SequenceNumber, seg.SequenceNumber)
		}
		assert.Len(t, req.Playlist.Playlist.Ads, 0)
	}
}

func TestOriginfootTimeout(t *testing.T) {
	timeout := 100 * time.Millisecond
	serverSleep := 20 * timeout
	fake := &originfootfakes.FakeOriginTenfootShim{}

	fake.SetPlaylistStub = func(ctx context.Context, _ *originfoot.SetPlaylistRequest) (*originfoot.SetPlaylistResponse, error) {
		select {
		case <-ctx.Done():
		case <-time.After(serverSleep):
		}
		return &originfoot.SetPlaylistResponse{}, nil
	}

	s := httptest.NewServer(originfoot.NewOriginTenfootShimServer(fake, nil, nil))
	defer s.Close()

	client := originfoot.NewOriginTenfootShimProtobufClient(s.URL, http.DefaultClient)

	o := &OriginFoot{
		base:     Noop(),
		client:   client,
		session:  &originctl.Session{Id: "1"},
		metadata: testMetadata,
		timeout:  timeout,
	}

	quality := "chunked"
	playlistPath := ""
	playlistType := "live"
	adBreaks := []*hlsext.AdBreakRequest{}
	futureSegments := []*hls.Segment{}

	start := time.Now()
	err := o.PostPlaylist(testPlaylist, quality, playlistPath, playlistType, adBreaks, futureSegments)
	timeTaken := time.Since(start)
	// We always log errors, but don't return them.
	assert.NoError(t, err)

	assert.True(t, timeTaken >= timeout, "shouldn't quit before the timeout")
	assert.True(t, timeTaken < 2*timeout, "shouldn't take way longer than the timeout")
}
