package tenfoot

import (
	"fmt"
	"math"
	"path/filepath"
	"time"

	"github.com/golang/protobuf/ptypes"
	"github.com/satori/go.uuid"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/m3u8"
	proto_common "code.justin.tv/video/protocols/common"
	"code.justin.tv/video/protocols/hls"
	"code.justin.tv/video/protocols/hlsext"
	"code.justin.tv/video/protocols/tenfoot"
)

// XXX: Variant of a function borrowed from pkg/m3u8/types.go.
func floatMsec(f float64) time.Duration {
	seconds := math.Floor(f)
	return (time.Second * time.Duration(seconds/1000.0)) + (time.Nanosecond * time.Duration(1e9*(f-seconds)))
}

// TranscodeMetadata describes the stream(s) that this instance of gotranscoder is producing.
type TranscodeMetadata struct {
	StartTimestamp        time.Time
	TargetSegmentDuration time.Duration
	TranscodeProfile      string
	Qualities             []string
	Channel               string
	UsherStreamID         int64
	SessionID             uuid.UUID
	ResourcePath          string

	OriginHostname string // e.g. "hls-abcdef"; this beomes the first path element in the OriginResourcePath.
}

// XXX: Should this move to video/protocols/tenfoot?
func (m *TranscodeMetadata) getPlaylistType(name string) (tenfoot.NewPlaylist_Type, error) {
	switch name {
	case "live":
		return tenfoot.NewPlaylist_M3U8_LIVE, nil
	case "history":
		return tenfoot.NewPlaylist_M3U8_HISTORY, nil
	// N.B.: These seem to be unused, but are defined in the video/protocols/tenfoot spec.
	// case "dvr":
	// 	return tenfoot.NewPlaylist_M3U8_DVR, nil
	// case "vod":
	// 	return tenfoot.NewPlaylist_M3U8_VOD, nil
	default:
		return tenfoot.NewPlaylist_UNKNOWN, fmt.Errorf("unknown playlist type: %v", name)
	}
}

func (m *TranscodeMetadata) buildStreamSubmessage(quality string) *proto_common.Stream {
	return &proto_common.Stream{
		Quality:            quality, // will be empty (~= not set) in Session message
		BroadcastId:        m.SessionID.Bytes(),
		UsherStreamId:      m.UsherStreamID,
		Channel:            m.Channel,
		OriginResourcePath: filepath.Join(m.OriginHostname, m.ResourcePath),
	}
}

// buildSessionMessage generates and returns a TranscodeSession message.  This message must be sent exactly once and
// it must be the first message sent by a newly-connected transcoder.
func (m *TranscodeMetadata) buildSessionMessage() (*tenfoot.TranscoderMessage, error) {

	startTimestampProto, err := ptypes.TimestampProto(m.StartTimestamp)
	if err != nil {
		return nil, fmt.Errorf("failed to generate timestamp: %v", err)
	}

	outerMsg := &tenfoot.TranscoderMessage{
		Timestamp: startTimestampProto,
		Stream:    m.buildStreamSubmessage(""),
		Msg: &tenfoot.TranscoderMessage_TranscodeSession{TranscodeSession: &tenfoot.TranscodeSession{
			Profile:   m.TranscodeProfile,
			Qualities: m.Qualities,
			Start:     startTimestampProto,
			TargetSegmentDuration: ptypes.DurationProto(m.TargetSegmentDuration),
		}}}

	return outerMsg, nil
}

// buildPlaylistMessage generates and returns a NewPlaylist message.
func (m *TranscodeMetadata) buildPlaylistMessage(playlist m3u8.Playlist, creation time.Time, quality, path, playlistTypeName string, adBreaks []*hlsext.AdBreakRequest) (*tenfoot.TranscoderMessage, error) {
	msgTimestamp, err := ptypes.TimestampProto(time.Now())
	if err != nil {
		return nil, fmt.Errorf("failed to generate timestamp: %v", err) // TODO: use error wrapping?
	}
	stream := m.buildStreamSubmessage(quality)
	// tenfoot should not receive any future segments for the purposes of this PR
	tenfootPlaylist, err := m.buildPlaylist(playlist, creation, quality, path, playlistTypeName, adBreaks, []*hls.Segment{})
	if err != nil {
		return nil, err
	}
	outerMsg := &tenfoot.TranscoderMessage{
		Timestamp: msgTimestamp,
		Stream:    stream,
		Msg:       &tenfoot.TranscoderMessage_Playlist{Playlist: tenfootPlaylist},
	}
	return outerMsg, nil
}

// buildSegmentMessage generates and returns a NewSegment message.
func (m *TranscodeMetadata) buildSegmentMessage(seg *avdata.Segment) (*tenfoot.TranscoderMessage, error) {
	quality := seg.Label

	msgTimestamp, err := ptypes.TimestampProto(time.Now())
	if err != nil {
		return nil, fmt.Errorf("failed to generate timestamp: %v", err)
	}

	outerMsg := &tenfoot.TranscoderMessage{
		Timestamp: msgTimestamp,
		Stream:    m.buildStreamSubmessage(quality),
		Msg: &tenfoot.TranscoderMessage_Segment{Segment: &tenfoot.NewSegment{
			Path: filepath.Join(m.ResourcePath, quality, seg.SegmentName),
			Segment: &hls.Segment{
				Uri:            seg.SegmentName,
				SequenceNumber: seg.SegmentNumber,
				Duration:       ptypes.DurationProto(floatMsec(seg.Duration)),
				Discontinuity:  false, // XXX: is this correct?
			},
		}}}

	return outerMsg, nil
}

func (m *TranscodeMetadata) buildPlaylist(playlist m3u8.Playlist, creation time.Time, quality, path, playlistTypeName string, adBreaks []*hlsext.AdBreakRequest, futureSegments []*hls.Segment) (*tenfoot.NewPlaylist, error) {
	playlistType, err := m.getPlaylistType(playlistTypeName)
	if err != nil {
		return nil, err
	}

	hlsPlaylist, err := playlist.ToHLSProto(creation)
	if err != nil {
		return nil, err
	}
	stream := m.buildStreamSubmessage(quality)

	// Per `video/protocols/tenfoot`, FutureSegments must be empty, and the Stream
	// field must match the top-level message's Stream field.
	hlsExtPlaylist := &hlsext.ExtendedPlaylist{
		Version:        0,
		Stream:         stream,
		Playlist:       hlsPlaylist,
		Ads:            adBreaks,
		FutureSegments: futureSegments,
		// TODO: Bitrate should be populated here.
	}
	return &tenfoot.NewPlaylist{
		Path:       path,
		Type:       playlistType,
		WindowSize: int64(playlist.WindowSize),
		Playlist:   hlsExtPlaylist,
	}, nil
}
