package twitchtranscoder

import (
	"bytes"
	"encoding/json"
	"log"
	"os"
	"os/exec"
	"strings"
	"time"

	"fmt"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/statsd"
)

// Twitch Transcoder Probe flags
//TODO - Move the ProbeInterval into a config
const (
	FlagProbeChannel  string = "--probe"
	FlagProbeDuration string = "--probeDuration"
	ProbeInterval     string = "2"
	ProbeResultPrefix string = "twitch_probe_result"
	ProbeErrorPrefix  string = "probe.error"
)

var probeTimeOut time.Duration

// SetProbeTimeoutDuration specifies the duration we should wait before retuning a probe timeout error
func SetProbeTimeoutDuration(timeout time.Duration) {
	probeTimeOut = timeout
}

// RunTwitchProbe probes the channel and return the data
func RunTwitchProbe(channel string, ttPath string) string {
	if channel == "" {
		return ""
	}

	var args []string

	args = append(args, FlagProbeChannel)
	args = append(args, channel)

	args = append(args, FlagProbeDuration)
	args = append(args, ProbeInterval)
	cmd := exec.Command(ttPath, args...)

	// Pipe routes
	cmd.Stdin = os.Stdin
	var outb, errb bytes.Buffer
	cmd.Stdout = &outb
	cmd.Stderr = &errb
	if err := cmd.Start(); err != nil {
		log.Printf(err.Error())
		return ""
	}
	//default timeout of 13 seconds for now
	timer := time.AfterFunc(probeTimeOut, func() {
		_ = cmd.Process.Kill()
	})
	err := cmd.Wait()
	if err != nil {
		errOp := errb.String()
		timer.Stop()
		log.Println("[PROBE]Error in running command:", err)
		errOp = strings.Replace(errOp, ":", "_", -1)
		statsd.Inc(fmt.Sprintf("%s.%s", ProbeErrorPrefix, strings.Replace(errOp, " ", "_", -1)), 1, 1)
		return ""
	}
	timer.Stop()

	op := outb.String()
	log.Println("[PROBE]Command output:", op)
	er := errb.String()

	if er == "" {
		log.Println("[PROBE]Error from running probe command , no output received:")
		return ""
	}
	return parseOutput(er)
}

func parseOutput(pOut string) string {
	temp := strings.Split(pOut, "\n")
	for _, line := range temp {
		if strings.Contains(line, ProbeResultPrefix) {
			return line
		}
	}
	return ""
}

// ParseProbeOutput deserializes probe output into the appropriate avdata type
func ParseProbeOutput(pOut string) (avdata.ProbeResult, bool) {
	if pOut == "" {
		return avdata.ProbeResult{}, true
	}
	var res avdata.ProbeResult
	err := json.Unmarshal([]byte(pOut), &res)
	if err != nil {
		log.Println("[PROBE]Error in unmarshalling Probe output", err)
		return avdata.ProbeResult{}, true
	}

	//Check if the error string is set in the probe output , log it for metrics and then return
	errMsg := res.ProbeAV.Error
	if errMsg != "" {
		log.Println("[PROBE]Probe returned an error message", errMsg)
		errMsg = strings.Replace(errMsg, ":", "_", -1)
		statsd.Inc(fmt.Sprintf("%s.%s", ProbeErrorPrefix, strings.Replace(errMsg, " ", "_", -1)), 1, 1)
		//Return ProbeResult with defaults value set
		return avdata.ProbeResult{}, true
	}

	if res.ProbeAV.Video.Height == 0 || res.ProbeAV.Video.Width == 0 {
		return avdata.ProbeResult{}, true
	}

	return res, false
}
