/*
Package twitchtranscoder wraps the Twitch Transcoder binary.

This executes twitch transcoder and routes stdin,err,out. back into the worker
*/
package twitchtranscoder

import (
	"bufio"
	"fmt"
	"io"
	"log"
	"os"
	"os/exec"
	"strconv"
	"syscall"
	"time"

	"strings"

	"code.justin.tv/video/gotranscoder/pkg/statsd"
)

// Twitch Transcoder flags
const (
	FlagHlsValidate     string = "--hls_validate"
	FlagChannel         string = "--channel"
	FlagEncoderConfig   string = "--encoder_config"
	FlagHlsUrlBase      string = "--hls_urlBase"
	FlagRandChars       string = "--rand_chars"
	RtmpUrlOpenError    string = "twitchtranscoder.rtmp_url_open_error"
	RtmpErrorPrefix     string = "Cound not open stream"
	TranscoderlogPrefix string = "twitchtranscoder.log"
)

// RuntimeSettings for TwitchTranscoder
type RuntimeSettings struct {
	HlsValidation     bool
	HlsUrlBase        string
	Channel           string
	EncoderConfig     string
	ListenerPort      int
	Path              string
	OutputJSONParser  func([]byte)
	OutputDebugParser func(string)
	cmdStdin          io.WriteCloser

	DisableObfuscation bool
}

// Setup the executable command based on the parameters passed
// This params mimic the ones required by TwitchTranscoder
func setupCommand(cfg *RuntimeSettings) (exec.Cmd, io.WriteCloser, error) {
	var args []string

	if cfg.HlsValidation {
		args = append(args, FlagHlsValidate)
		args = append(args, strconv.FormatBool(cfg.HlsValidation))
	}

	if cfg.HlsUrlBase != "" {
		args = append(args, FlagHlsUrlBase)
		args = append(args, cfg.HlsUrlBase)
	}

	if cfg.Channel != "" {
		args = append(args, FlagChannel)
		args = append(args, cfg.Channel)
	}

	if cfg.EncoderConfig != "" {
		args = append(args, FlagEncoderConfig)
		args = append(args, cfg.EncoderConfig)
	}

	if cfg.DisableObfuscation {
		args = append(args, FlagRandChars)
		args = append(args, strconv.FormatBool(false))
	}

	cmd := exec.Command(cfg.Path, args...)

	// Pipe routes
	stdin, _ := cmd.StdinPipe()
	stdout, _ := cmd.StdoutPipe()
	stderr, _ := cmd.StderrPipe()

	go ScanLibAvOutput(cfg, stdout, os.Stdout, "[StdOut]")
	go ScanLibAvOutput(cfg, stderr, os.Stderr, "[StdErr]")

	return *cmd, stdin, nil
}

// ScanLibAvOutput will parse output from libavtwitch
// and forward to the appropiate Os out (stdErr or stdOut)
// Forwarding is needed since this is how the python worker reads output
func ScanLibAvOutput(cfg *RuntimeSettings, reader io.Reader, out *os.File, label string) {
	scanner := bufio.NewScanner(reader)

	for scanner.Scan() {
		line := scanner.Text()
		cfg.OutputDebugParser(line)
		cfg.OutputJSONParser(scanner.Bytes())

		// Write to OS.StdXX - this is for backward compatibility on the python worker
		//fmt.Fprintf(out, "[LIBAV MESSAGE] %s %s \n", label, line) // for debugging purposes
		fmt.Fprintf(out, "%s\n", line)
		if strings.Contains(line, RtmpErrorPrefix) {
			statsd.Inc(RtmpUrlOpenError, 1, 1)
		}

	}
	log.Println("[LIBAVSCANNER] scanner output stopped")
}

// Run Twitchtranscoder and wait for it's completion.
func (cfg *RuntimeSettings) Run() error {
	var cmd, stdin, err = setupCommand(cfg)
	log.Printf("Starting TwitchTrascoder with: %+v", cmd)
	var runtimeStart = time.Now()

	if err != nil {
		log.Println("Failed to setup command:", err)
		return err
	}

	cfg.cmdStdin = stdin
	addPdeathsig(&cmd, syscall.SIGKILL)

	err = cmd.Run()

	if err != nil {
		log.Printf("failed to launch : %v", err)
	}

	// Log the error exit codes.
	msg, _ := err.(*exec.ExitError)
	if msg != nil {
		exitCode := msg.Sys().(syscall.WaitStatus).ExitStatus()
		statsd.Inc(fmt.Sprintf("twitchtranscoder.exit_code.%d", exitCode), 1, 1)
	} else {
		statsd.Inc("twitchtranscoder.exit_code.unknown", 1, 1)
	}

	statsd.Timing("twitchtranscoder.execution_time", time.Since(runtimeStart), 1.0)
	return err
}

// Start TwitchTranscoder with cmd.start , non blocking
func (cfg *RuntimeSettings) Start() (*exec.Cmd, error) {
	var cmd, stdin, err = setupCommand(cfg)
	if err != nil {
		log.Println("Failed to setup command:", err)
	}

	cfg.cmdStdin = stdin
	addPdeathsig(&cmd, syscall.SIGKILL)

	log.Println(cmd)
	err = cmd.Start()

	if err != nil {
		log.Printf("failed to launch : %v", err)
		return nil, err
	}

	return &cmd, nil
}

// Wait for cmd execution
// Plan is to use this in the following versions - currently
// when wait is called from outside this method pipes will close
// and terminate Twitchtranscoder process loop.
func (cfg *RuntimeSettings) Wait(cmd *exec.Cmd) error {
	err := cmd.Wait()

	if err != nil {
		log.Println("failed waiting for command", err)
	}

	if msg, ok := err.(*exec.ExitError); ok { // there is error code
		exitCode := msg.Sys().(syscall.WaitStatus).ExitStatus()
		statsd.Inc(fmt.Sprintf("twitchtranscoder.exit_code.%d", exitCode), 1, 1)
	}

	return err

}

// WriteToStdin writes a raw strings to TwitchTranscoder's stdin
func (cfg *RuntimeSettings) WriteToStdin(s string) error {
	_, err := cfg.cmdStdin.Write([]byte(s))
	return err
}
