package twitchtranscoder

import (
	"bytes"
	"fmt"
	"log"
	"os"
	"os/exec"
	"strconv"
	"sync"
	"testing"

	"github.com/stretchr/testify/assert"
)

// Settings to build mock executable prior tests
var buildArtifactPath = "./pkg/twitchtranscoder/_build_artifacts/mock_exec"

var mockCompilePath = "../../cmd/mocktwitchtranscoder"

// Setup Data

var testValidationModeArgs = "--hls_validate true --channel test\n"
var testTranscoderModeArgs = "--encoder_config ../test/data/TwitchTranscoder/config.json\n"

var testRuntimeSettings = RuntimeSettings{
	HlsValidation:     true,
	Channel:           "test",
	EncoderConfig:     "../test/data/TwitchTranscoder/config.json",
	Path:              buildArtifactPath,
	OutputJSONParser:  parseJSONLine,
	OutputDebugParser: parseDebugLine,
}

var testInvalidRuntimeSettings = RuntimeSettings{
	HlsValidation:     true,
	Channel:           "test",
	EncoderConfig:     "config_test.json",
	Path:              "/doesnt_exist",
	OutputJSONParser:  parseJSONLine,
	OutputDebugParser: parseDebugLine,
}

var testValidationMode = RuntimeSettings{
	HlsValidation:     true,
	Channel:           "test",
	Path:              buildArtifactPath,
	OutputJSONParser:  parseJSONLine,
	OutputDebugParser: parseDebugLine,
}

var testTranscodeMode = RuntimeSettings{
	EncoderConfig:     "../test/data/TwitchTranscoder/config.json",
	Path:              buildArtifactPath,
	OutputJSONParser:  parseJSONLine,
	OutputDebugParser: parseDebugLine,
}

var testStdinWritesMode = &RuntimeSettings{
	EncoderConfig:     "../test/data/TwitchTranscoder/config.json",
	Path:              buildArtifactPath,
	OutputJSONParser:  parseJSONLine,
	OutputDebugParser: parseDebugLine,
}

var testExitCodesSettings = RuntimeSettings{
	HlsValidation:     true,
	Channel:           "exitcode",
	EncoderConfig:     "../test/data/TwitchTranscoder/config.json",
	Path:              buildArtifactPath,
	OutputJSONParser:  parseJSONLine,
	OutputDebugParser: parseDebugLine,
}

func TestStartWaitTranscoder(t *testing.T) {
	t.Skip("Skipping due to race condition")
	assert := assert.New(t)

	cmd, err := testRuntimeSettings.Start()
	assert.Nil(err)
	assert.NotNil(cmd)

	err = testRuntimeSettings.Wait(cmd)
	assert.Nil(err)
}

func parseJSONLine(data []byte) {
	fmt.Println(data)
}

func parseDebugLine(line string) {
	fmt.Println(line)
}

func TestRunTranscoder(t *testing.T) {
	t.Skip("Race condition")
	assert := assert.New(t)
	err := testRuntimeSettings.Run()
	assert.Nil(err)
}

// Test Negative Edge Cases
func TestInvalidBinaries(t *testing.T) {
	t.Skip("Race condition")
	assert := assert.New(t)

	err := testInvalidRuntimeSettings.Run()
	assert.NotNil(err)

	_, err = testInvalidRuntimeSettings.Start()
	assert.NotNil(err)
}

// Test the setup of a command
func TestSetupCommand(t *testing.T) {
	assert := assert.New(t)
	var cmd, _, err = setupCommand(&testRuntimeSettings)

	assert.Nil(err)

	assert.Equal(testRuntimeSettings.Path, cmd.Args[0])
	assert.Equal(FlagHlsValidate, cmd.Args[1])
	assert.Equal(strconv.FormatBool(testRuntimeSettings.HlsValidation), cmd.Args[2])
	assert.Equal(FlagChannel, cmd.Args[3])
	assert.Equal(testRuntimeSettings.Channel, cmd.Args[4])
	assert.Equal(FlagEncoderConfig, cmd.Args[5])
	assert.Equal(testRuntimeSettings.EncoderConfig, cmd.Args[6])
}

// Test the runtime settings
func TestRuntimeSettings(t *testing.T) {
	t.Skip("Race condition")
	assert := assert.New(t)
	var transcodeMode, _, errT = setupCommand(&testTranscodeMode)
	var validationMode, _, errV = setupCommand(&testValidationMode)

	var bTranscodeOutput, bValidationOutput bytes.Buffer
	transcodeMode.Stdout = &bTranscodeOutput
	validationMode.Stdout = &bValidationOutput

	errTM := transcodeMode.Run()
	errVM := validationMode.Run()

	assert.Nil(errT)
	assert.Nil(errV)
	assert.Nil(errTM)
	assert.Nil(errVM)

	assert.Equal(testValidationModeArgs, string(bValidationOutput.Bytes()))
	assert.Equal(testTranscoderModeArgs, string(bTranscodeOutput.Bytes()))

}

func TestStdinWrites(t *testing.T) {
	t.Skip("Race condition")
	assert := assert.New(t)

	cmd, errS := testStdinWritesMode.Start()
	errWr := testStdinWritesMode.WriteToStdin("fake_command")
	errW := testStdinWritesMode.Wait(cmd)

	assert.Nil(errS)
	assert.Nil(errWr)
	assert.Nil(errW)

}

// Do a validation of different TwitchTranscoder exit modes
func TestExit(t *testing.T) {
	t.Skip("Skipping due to a race condition")
	assert := assert.New(t)

	testExitCodesSettings.EncoderConfig = "0"
	cmd, _, err := setupCommand(&testExitCodesSettings)
	assert.Nil(err)
	assert.NoError(cmd.Run())

	testExitCodesSettings.EncoderConfig = "200"
	cmd, _, err = setupCommand(&testExitCodesSettings)
	assert.Nil(err)
	assert.Error(cmd.Run())

	testExitCodesSettings.EncoderConfig = "1"
	cmd, _, err = setupCommand(&testExitCodesSettings)
	assert.Nil(err)
	assert.Error(cmd.Run())
}

// Perform the build step of Mock twitchtranscoder and execute the tests
func TestMain(m *testing.M) {
	// build the required mock
	var mu = &sync.Mutex{}

	mu.Lock()
	log.Println("Building mock twitch transcoder")
	_ = os.Remove(buildArtifactPath)
	cmd := exec.Command("go", "build", "-o", buildArtifactPath, mockCompilePath)
	err := cmd.Run()

	if err != nil {
		log.Println("Error building mock executable", err)
	}

	//Build twitch probe mock executable
	log.Println("Building mock twitch transcoder probe")
	_ = os.Remove(mockCompileProbePath)
	cmd = exec.Command("go", "build", "-o", twitchTranscoderProbePath, mockCompileProbePath)
	err = cmd.Run()

	if err != nil {
		log.Println("Error building mock probe executable", err)
	}

	mu.Unlock()

	m.Run()
}
