package usher

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"net/url"
)

// Settings contains the usher configuration
type Settings struct {
	BaseURL   string
	Stateless bool
}

// Vod id type
type Vod struct {
	VodID int `json:"vod_id"`
}

// Post to Usher Api
func (cfg *Settings) Post(relativePath string, data interface{}) ([]byte, error) {
	if cfg.Stateless {
		log.Printf("Gotranscoder Stateless Mode: Skipping usher call %s\n", relativePath)
		return nil, nil
	}

	var bodyReader io.Reader

	path, err := cfg.APIPath(relativePath, "")

	if data != nil {
		body, err := json.Marshal(data)
		if err != nil {
			return nil, err
		}
		log.Printf("[Usher API] Marshalled Json body for post: %s", body)
		bodyReader = bytes.NewBuffer(body)
	}

	client := &http.Client{}

	req, err := http.NewRequest("POST", path, bodyReader)

	req.Header.Add("Content-Type", `application/json`)
	req.Header.Add("x-jtv-authorization", `staff`)

	response, err := client.Do(req)

	if err != nil {
		log.Printf("[Usher API] Error Posting to %s - %s\n", path, err)
		return nil, err
	}

	if response.StatusCode != http.StatusOK {
		err := response.Body.Close()
		return nil, fmt.Errorf("[Usher API] Failed to perform usher request.\n Status[%d].\n Path [%s].\n Response [%v],\n body close [%s]\n", response.StatusCode, relativePath, response, err)
	}

	contents, err := ioutil.ReadAll(response.Body)
	if err != nil {
		log.Printf("[Usher API] Failed to get body of response %s\n", err)
		return nil, err
	}

	return contents, nil
}

// Get from Usher Api
func (cfg *Settings) Get(relativePath string, query string) ([]byte, error) {
	// TODO: exponential backoff retries
	path, err := cfg.APIPath(relativePath, query)
	if err != nil {
		log.Printf("Failed to generate an Usher Api path for the relative path %s", relativePath)
		return nil, err
	}

	response, err := http.Get(path)
	if err != nil {
		return nil, err
	}

	if response.StatusCode != http.StatusOK {
		err := response.Body.Close()
		return nil, fmt.Errorf("Failed to perform usher request. Status[%d]. Path [%s]. Response [%v], Body Close [%s]", response.StatusCode, relativePath, response, err)
	}

	contents, err := ioutil.ReadAll(response.Body)
	if err != nil {
		log.Println("Failed to get body of response", err)
		return nil, err
	}

	return contents, nil
}

// APIPath creates a full usher url
func (cfg *Settings) APIPath(path string, query string) (string, error) {

	u, err := url.Parse(cfg.BaseURL)
	if err != nil {
		return "", err
	}

	u.Path = path
	u.RawQuery = query

	return u.String(), nil
}

// GetShowChannel shows the stream and transcode information for a live channel.
// Queries by channel name, and returns the contents of HLS_Transcode in usher
// particularly useful to get the transcode id for a stream.
func (cfg *Settings) GetShowChannel(channel string) (*HlsTranscode, error) {
	response, err := cfg.Get(fmt.Sprintf(`/hls_transcode/show_channel/%s.json`, channel), "")
	if err != nil {
		log.Println("Failed to get the channel information", err)
		return nil, err
	}

	var info = make([]HlsTranscode, 0)
	err = json.Unmarshal(response, &info)
	if err != nil {
		log.Printf("Failed parse json output: %s - %s", response, err)
		return nil, err
	}
	return &info[0], nil
}

// GetHlsTranscode returns metadata for a transcode
// Queries by Transcode ID and returns the contents of HLS_Transcode in usher
// AND some stream info
func (cfg *Settings) GetHlsTranscode(transcodeID int) (*HlsTranscode, error) {
	response, err := cfg.Get(fmt.Sprintf(`/hls_transcode/show/%d.json`, transcodeID), "")
	if err != nil {
		log.Println("Failed to get hls transcode information", err)
		return nil, err
	}

	var transcode = make([]HlsTranscode, 0)
	err = json.Unmarshal(response, &transcode)
	if err != nil {
		log.Printf("Failed parse json output: %s - %s", response, err)
		return nil, err
	}
	return &transcode[0], nil
}

// MarkHlsStreamReady sets the status of a stream to ready when things are in a good state
// and the channel can be viewed
func (cfg *Settings) MarkHlsStreamReady(streamid int) error {
	_, err := cfg.Post(fmt.Sprintf(`/hls_transcode/ready/%d.json`, streamid), nil)
	if err != nil {
		log.Println("Failed to mark the stream as ready in usher", err)
		return err
	}

	return nil
}

// MarkLVSHlsStreamReady sets the status of a stream to ready when things are in a good state
// and the channel can be viewed
func (cfg *Settings) MarkLvsStreamReady(streamid int) error {
	_, err := cfg.Post(fmt.Sprintf(`/hls_transcode/lvs_ready/%d.json`, streamid), nil)
	if err != nil {
		log.Println("Failed to mark the lvs stream as ready in usher", err)
		return err
	}

	return nil
}

// SendRabbitJob posts a message to usher to insert a job into rabbit mq.
func (cfg *Settings) SendRabbitJob(key string, vodID int) error {
	vod := Vod{VodID: vodID}
	_, err := cfg.Post(fmt.Sprintf(`/job/send/%s.json`, key), vod)
	if err != nil {
		log.Println("Failed to send job request to usher", err)
		return err
	}

	return nil
}

// StreamInfo returns the metadata for a stream.
// userful to retrieve the channel's broadcaster and general information.
func (cfg *Settings) StreamInfo(channel string) ([]StreamInfo, error) {
	log.Printf("Requesting Stream Info For: %s\n", channel)
	response, err := cfg.Get(`/stream/list.json`, fmt.Sprintf(`channel=%s`, channel))
	if err != nil {
		log.Println("Failed to get the stream information", err)
		return nil, err
	}

	var streams = make([]StreamInfo, 0)
	err = json.Unmarshal(response, &streams)
	if err != nil {
		log.Printf("Failed parse json output: %s - %s", response, err)
		return nil, err
	}
	log.Printf("Received Stream Info: %+v\n", channel)
	return streams, nil
}

// KillHlsTranscode marks a transcode as terminated on usher DB
func (cfg *Settings) KillHlsTranscode(transcodeID int) error {
	_, err := cfg.Post(fmt.Sprintf(`/hls_transcode/kill/%d.json`, transcodeID), nil)
	if err != nil {
		log.Println("Failed to kill the transcode", err)
		return err
	}

	return nil
}

// UpdateStreamProperties sets the audio and video codec information in usher
// which is used to populate the "Excellent, medium, bad,"
// stream configuration quality labels in the video dashboard.
// http://www.twitch.tv/[channel]/dashboard
//
// TODO: Fix endpoint in usher to receive a channel name not a channel property ID.
func (cfg *Settings) UpdateStreamProperties(propertyID int, codecInfo interface{}) error {
	_, err := cfg.Post(fmt.Sprintf(`/channel_properties/update/%d.json`, propertyID), codecInfo)
	if err != nil {
		log.Printf("Failed to update the stream properties in usher: %s, %+v \n", err, codecInfo)
		return err
	}

	return nil
}

// UpdateHlsEntry pushes an update to the hls_transcode table.
// This is used to keep track of the transcode state.
func (cfg *Settings) UpdateHlsEntry(transcodeID int, hlsEntry interface{}) error {
	log.Printf("HLS ENTRY: %+v\n", hlsEntry)

	_, err := cfg.Post(fmt.Sprintf(`/hls_transcode/update/%d.json`, transcodeID), hlsEntry)
	if err != nil {
		log.Printf("Failed to update the HLS transcode entry in usher: %s, %+v \n", err, hlsEntry)
		return err
	}

	return nil
}

// UpdateHlsTranscode pushes an update to the hls_transcode table.
// This is used to keep track of the transcode state.
func (cfg *Settings) UpdateHlsTranscode(transcodeID int, hlsEntry *HlsTranscode) error {
	return cfg.UpdateHlsEntry(transcodeID, hlsEntry)
}

// GetStreamCodec returns the current ingested stream properties.
// used to retrieve a property ID in order to post continous updates on the stats of
// the ingested stream
func (cfg *Settings) GetStreamCodec(channel string) (*StreamCodec, error) {
	response, err := cfg.Get(fmt.Sprintf(`/channel_properties/show/%s.json`, channel), ``)
	if err != nil {
		log.Printf("Failed to get the channel properties from usher: %s, %s \n", channel, err)
		return nil, err
	}

	var properties = make([]StreamCodec, 0)
	err = json.Unmarshal(response, &properties)
	if err != nil {
		log.Printf("Failed parse json output: %s - %s", response, err)
		return nil, err
	}

	return &properties[0], nil
}

// GetDbOption returns the value of an specified DB Option
func (cfg *Settings) GetDbOption(option string) (*DBOption, error) {
	response, err := cfg.Get(fmt.Sprintf(`/dboption/get/%s.json`, option), ``)
	if err != nil {
		log.Println("Failed to get the stream information", err)
		return nil, err
	}

	var optionValues = make([]DBOption, 0)
	err = json.Unmarshal(response, &optionValues)
	if err != nil {
		log.Printf("Failed parse json output: %s - %s", response, err)
		return nil, err
	}
	return &optionValues[0], nil
}
