package vod

// This file contains mock utilities used in internal tests

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"os"
	"time"

	"code.justin.tv/video/gotranscoder/pkg/notify"
	"code.justin.tv/video/gotranscoder/pkg/usher"
	"github.com/gorilla/mux"
)

type Mocks3Conn struct {
	Connection
}

func (c *Mocks3Conn) Put(bucket, objName string, data []byte, contentType string) error {
	return nil
}
func (c *Mocks3Conn) PutNoRetry(bucket, objName string, data []byte, contentType string) error {
	return nil
}

func NewMockS3Conn() Connection {
	return &Mocks3Conn{}
}

type mockOriginSettings struct {
	enable  bool
	code    int
	timeout bool
}

type testServer struct {
	server *httptest.Server
	Calls  map[string]int
}

func NewMockOriginServer(settings mockOriginSettings) *testServer {

	newServer := testServer{
		Calls: make(map[string]int),
	}
	code := settings.code
	if code == 0 {
		code = 200
	}
	router := mux.NewRouter()
	router.HandleFunc("/streams/{id}/thumb/{thumbname}", func(w http.ResponseWriter, r *http.Request) {
		newServer.Calls["ReadThumbnail"] += 1
		if settings.timeout {
			time.Sleep(2 * time.Millisecond)
		}
		w.WriteHeader(code)
	})

	router.HandleFunc("/streams/{id}/{variant}/{segname}", func(w http.ResponseWriter, r *http.Request) {
		newServer.Calls["ReadSegment"] += 1
		if settings.timeout {
			time.Sleep(2 * time.Millisecond)
		}
		w.WriteHeader(code)
	})

	newServer.server = httptest.NewServer(router)

	return &newServer
}

func NewMockVinylServer() *testServer {
	newServer := testServer{
		Calls: make(map[string]int),
	}

	router := mux.NewRouter()
	router.HandleFunc("/v1/vods/past_broadcast", func(w http.ResponseWriter, r *http.Request) {
		newServer.Calls["register"] += 1

		w.WriteHeader(200)
		w.Header().Set("Content-Type", "application/json")
		type pastBroadcast struct {
			ID int `json:"id"`
		}

		var jsonResp struct {
			PastBroadcast pastBroadcast `json:"past_broadcast"`
		}

		jsonResp.PastBroadcast.ID = 1234
		json.NewEncoder(w).Encode(jsonResp)
	})

	router.HandleFunc("/v1/vods/{vodid}/thumbnails", func(w http.ResponseWriter, r *http.Request) {
		newServer.Calls["thumbnails"] += 1
		w.WriteHeader(200)

	})

	router.HandleFunc("/v1/vods/{vodid}/", func(w http.ResponseWriter, r *http.Request) {
		newServer.Calls["finalize"] += 1
		w.WriteHeader(200)
	})

	newServer.server = httptest.NewServer(router)
	return &newServer
}

// NewPusherWithMockDefault creates a pusher and
// corresponding mock server with settings good for testing
func NewPusherWithMockDefault(queueSize, chunkFactor int, origSettings mockOriginSettings) (*Pusher, func(), *testServer, *testServer) {
	var originServer *testServer
	if origSettings.enable {
		originServer = NewMockOriginServer(origSettings)
	}
	vinylServer := NewMockVinylServer()

	settings := PusherSettings{
		ChannelName:      "TrumpSC",
		ChannelID:        123455667,
		BroadcastID:      14366239750,
		VodKey:           "TrumpSC_14366239750_1234212",
		TranscodePath:    os.TempDir(),
		QueueSize:        queueSize,
		NotifyQueueSize:  100,
		NumThumbnails:    4,
		ChunkFactor:      chunkFactor,
		VinylHost:        vinylServer.server.URL[7:],
		PlaylistFileName: "index-dvr.m3u8",
		Usher:            usher.Settings{BaseURL: "http://usher.ttvnw.net/"},
		Bucket:           "MyBucket",
		Conn:             NewMockS3Conn(),
		MinVodDurationSeconds: 10,
		Notifier:              &notify.NotifySettings{},
	}

	if origSettings.enable {
		settings.HlsUrlBase = fmt.Sprintf("%s/streams/1234", originServer.server.URL)
		if origSettings.timeout {
			settings.OriginTimeout = 500 * time.Microsecond
		}
	}

	qualities := []Quality{{
		Label:              "source",
		DisplayName:        "Source",
		PlaylistPreference: 0,
	}, {
		Label:              "medium",
		DisplayName:        "Medium",
		PlaylistPreference: 1,
	}}

	cleanup := func() {
		vinylServer.server.Close()
		if origSettings.enable {
			originServer.server.Close()
		}
	}
	return NewPusher(settings, qualities), cleanup, vinylServer, originServer
}
