package vod

import (
	"sync"

	"code.justin.tv/video/gotranscoder/pkg/m3u8"
)

// Playlist manages async operations on playlist
type Playlist struct {
	list []m3u8.Chunk
	lock sync.RWMutex
}

// Append adds new chunk to the playlist
func (p *Playlist) Append(c m3u8.Chunk) {
	p.lock.Lock()
	c.SequenceNumber = int64(len(p.list))
	p.list = append(p.list, c)
	p.lock.Unlock()
}

// Len returns length of the playlist with thread-safety
func (p *Playlist) Len() int {
	p.lock.Lock()
	defer p.lock.Unlock()

	return len(p.list)
}

// Copy creates a snapshot of the playlist
func (p *Playlist) Copy() []m3u8.Chunk {
	p.lock.RLock()
	defer p.lock.RUnlock()

	newList := []m3u8.Chunk{}
	for _, c := range p.list {
		newList = append(newList, c)
	}
	return newList
}

// MarkDiscontinuous marks a playlist discontinuous based on other renditions
func (p *Playlist) MarkDiscontinuous(seqNum int) float64 {
	p.lock.Lock()
	defer p.lock.Unlock()
	var lostDur float64

	if int(seqNum) < len(p.list) && p.list[seqNum].URL != m3u8.DiscontinuityURL {
		p.list[seqNum].URL = m3u8.DiscontinuityURL
		lostDur += p.list[seqNum].Duration
		p.list[seqNum].Duration = 0
	}
	return lostDur

}

// Get snapshot of the playlist
func (p *Playlist) Get() []m3u8.Chunk {
	return p.Copy()
}
