package cloudformation

import (
	"errors"
	"fmt"
	"os"

	"code.justin.tv/vodsvc/aws/awsconfig"
	"github.com/aws/aws-sdk-go/aws"
	cf "github.com/aws/aws-sdk-go/service/cloudformation"
	cfiface "github.com/aws/aws-sdk-go/service/cloudformation/cloudformationiface"
)

type CloudFormation interface {
	GetPhysicalResourceId(logicalResource string) (string, error)
	MustGetPhysicalResourceId(logicalResource string) string
}

func New(stack string) CloudFormation {
	awsconfig.Logger.Log("Using CloudFormation stack: ", stack)
	return &cloudFormation{
		client:    cf.New(awsconfig.NewSession()),
		logger:    awsconfig.Logger,
		StackName: stack,
	}
}

func NewFromEnv() CloudFormation {
	stackName := os.Getenv("ENV_STACK_NAME")
	if stackName == "" {
		awsconfig.Logger.Log("ENV_STACK_NAME environment variable not set")
		os.Exit(1)
	}
	return New(stackName)
}

type cloudFormation struct {
	logger    aws.Logger
	client    cfiface.CloudFormationAPI
	StackName string
}

func (c *cloudFormation) MustGetPhysicalResourceId(logicalResource string) string {
	id, err := c.GetPhysicalResourceId(logicalResource)
	if err != nil {
		panic(err)
	}
	return id
}

func (c *cloudFormation) GetPhysicalResourceId(logicalResource string) (string, error) {
	input := &cf.DescribeStackResourceInput{
		LogicalResourceId: aws.String(logicalResource),
		StackName:         aws.String(c.StackName),
	}
	out, err := c.client.DescribeStackResource(input)
	if err != nil {
		errmsg := "Couldn't retrieve physical resource name from CloudFormation"
		msg := fmt.Sprintf("%s; resource=%s, err=%s\n", errmsg, logicalResource, err)
		return "", errors.New(msg)
	}
	c.logger.Log("Found resource from CloudFormation: ", logicalResource+" -> ", *out.StackResourceDetail.PhysicalResourceId)
	return *out.StackResourceDetail.PhysicalResourceId, nil
}
