package api

import (
	"context"
	"encoding/json"
	"time"

	sts "code.justin.tv/vodsvc/sts/src/transcoder"
	"code.justin.tv/vodsvc/vhs/src/types"
	"github.com/aws/aws-lambda-go/events"
)

func (a *API) CompleteTranscode(ctx context.Context, e events.S3Event) {
	for _, r := range e.Records {
		uploadId := r.S3.Object.Key[0:36]

		manifestBytes, err := a.videoBucket.GetObject(r.S3.Object.Key)
		if err != nil {
			a.logger.Error("Error retrieving video manifest from S3: " + err.Error())
			a.failUpload(uploadId)
			return
		}

		var manifest sts.Video
		if err := json.Unmarshal(manifestBytes, &manifest); err != nil {
			a.logger.Error("Error unmarshalling manifest to JSON: " + err.Error())
			a.failUpload(uploadId)
			return
		}

		video := types.Video{
			Id:              uploadId,
			CreateTime:      time.Now().UTC(),
			PendingDeletion: false,
			S3Bucket:        r.S3.Bucket.Name,
			S3Prefix:        r.S3.Object.Key[0:73],
			VideoType:       types.TypeUpload,
		}

		video = a.transcoder.ParseManifest(video, manifest)

		if err := a.videoTable.CreateVideo(video); err != nil {
			a.logger.Errorf("Error creating video in Dynamo table; id=%v, err=%v", uploadId, err)
			a.failUpload(uploadId)
			return
		}

		if err := a.uploadTable.SetStatus(uploadId, types.StatusProcessed); err != nil {
			a.logger.Errorf("Error setting upload status; id=%v, err=%v",
				uploadId, err)
			a.failUpload(uploadId)
			return
		}
	}
}

func (a *API) failUpload(id string) {
	err := a.uploadTable.SetFailed(id, types.ErrInternalFailure)
	if err != nil {
		a.logger.Errorf("Error setting failed status in DynamoDB; id=%v, err=%v", id, err)
	}
}
