package api

import (
	"context"
	"encoding/base64"
	"fmt"
	"os"
	"strings"
	"time"
	"unicode"

	"code.justin.tv/vodsvc/aws/awsconfig"
	"code.justin.tv/vodsvc/httpserver"
	"code.justin.tv/vodsvc/vhs/src/types"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/kms"
	"github.com/aws/aws-sdk-go/service/sts"
)

type CreateUploadResponse struct {
	Upload      types.Upload `json:"upload"`
	Credentials Credentials  `json:"credentials"`
	S3URI       string       `json:"s3Uri"`
}

type Credentials struct {
	AccessKeyId     string `json:"accessKeyId"`
	SecretAccessKey string `json:"secretAccessKey"`
	SessionToken    string `json:"sessionToken"`
	Expiration      string `json:"expiration"`
}

const defaultUploadDuration = 12 * time.Hour / time.Second
const iamPolicy = `
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": ["s3:PutObject","s3:AbortMultipartUpload","s3:ListMultipartUploadParts"],
      "Resource": "arn:aws:s3:::%v/%v"
    }
  ]
}
`

func (a *API) CreateUpload(ctx context.Context, r events.APIGatewayProxyRequest) (interface{}, *httpserver.HTTPError) {
	var err error

	id := generateVideoId()
	s3prefix := generateS3Prefix(id)
	s3bucket := a.uploadBucket.BucketName()
	videoBucket := a.videoBucket.BucketName()
	sourceKey := sourceKey(s3prefix)

	upload, err := a.uploadTable.CreateUpload(id, s3bucket, s3prefix, videoBucket, sourceFile)
	if err != nil {
		return nil, httpserver.InternalServerError(err)
	}

	creds, err := a.getCreds(id, s3bucket, sourceKey)
	if err != nil {
		a.logger.Error("Error getting credentials: " + err.Error())
		return nil, httpserver.InternalServerError(err)
	}

	response := &CreateUploadResponse{
		Upload:      upload,
		Credentials: creds,
		S3URI:       s3URI(s3bucket, sourceKey),
	}

	return response, nil
}

func (a *API) getCreds(id, s3bucket, sourceKey string) (Credentials, error) {
	policy := squish(fmt.Sprintf(iamPolicy, s3bucket, sourceKey))

	accessKey := os.Getenv("FEDERATION_ACCESS")
	secretKey := os.Getenv("FEDERATION_SECRET")
	kmsClient := kms.New(awsconfig.Session)

	b64secret, err := base64.StdEncoding.DecodeString(secretKey)
	if err != nil {
		return Credentials{}, fmt.Errorf("Error base64 decoding secret key: %v", err)
	}

	decryptInput := &kms.DecryptInput{
		CiphertextBlob: []byte(b64secret),
	}

	decryptOutput, err := kmsClient.Decrypt(decryptInput)
	if err != nil {
		return Credentials{}, fmt.Errorf("Error decrypting secret key: %v", err)
	}

	awsCreds := credentials.NewStaticCredentials(accessKey, string(decryptOutput.Plaintext), "")
	config := aws.Config{Credentials: awsCreds}
	stsClient := sts.New(session.Must(session.NewSession(&config)))

	input := &sts.GetFederationTokenInput{
		DurationSeconds: aws.Int64(int64(defaultUploadDuration)),
		Name:            aws.String(id[0:32]),
		Policy:          aws.String(policy),
	}

	output, err := stsClient.GetFederationToken(input)
	if err != nil {
		return Credentials{}, err
	}

	creds := Credentials{
		AccessKeyId:     *output.Credentials.AccessKeyId,
		SecretAccessKey: *output.Credentials.SecretAccessKey,
		SessionToken:    *output.Credentials.SessionToken,
		Expiration:      output.Credentials.Expiration.Format(time.RFC3339),
	}

	return creds, nil
}

func squish(str string) string {
	return strings.Map(func(r rune) rune {
		if unicode.IsSpace(r) {
			return -1
		}
		return r
	}, str)
}
