package dynamodb

import (
	"time"

	"github.com/aws/aws-sdk-go/service/dynamodb"
	ddbiface "github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"

	"code.justin.tv/vodsvc/aws/awsconfig"
	"code.justin.tv/vodsvc/aws/cloudwatch"
	"code.justin.tv/vodsvc/vhs/src/types"
	log "go.uber.org/zap"
)

const ScanLimit = 100

type UploadTable interface {
	AddEtag(id string, part int, etag string, size int) error
	CreateUpload(id, s3Bucket, s3Prefix, videoBucket, sourceFile string) (types.Upload, error)
	GetUpload(id string) (types.Upload, error)
	SetStatus(id, status string) error
	SetFailed(id, reason string) error
	ListUploads(startKey string) ([]types.Upload, string, error)
	TableName() string
}

type VideoTable interface {
	CreateVideo(video types.Video) error
	UpdateVideo(video *types.Video) error
	GetVideo(id string) (types.Video, error)
	DeleteVideo(id string) error
	ListVideos(startKey string) ([]*types.Video, string, error)
	TableName() string
}

func New(table string, logger *log.SugaredLogger) *ddb {
	return &ddb{
		cloudwatch: cloudwatch.New(logger),
		client:     dynamodb.New(awsconfig.Session),
		table:      table,
		logger:     logger,
	}
}

func (d *ddb) TableName() string {
	return d.table
}

type ddb struct {
	cloudwatch cloudwatch.CloudWatch
	client     ddbiface.DynamoDBAPI
	logger     *log.SugaredLogger
	table      string
}

func now() time.Time {
	return time.Now().UTC()
}

func now3339() string {
	return now().Format(time.RFC3339)
}
