package dynamodb

import (
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"

	cw "code.justin.tv/vodsvc/aws/cloudwatch"
	"code.justin.tv/vodsvc/vhs/src/types"
)

func (d *ddb) SetStatus(id, status string) error {
	input := &dynamodb.UpdateItemInput{
		TableName:        aws.String(d.table),
		Key:              map[string]*dynamodb.AttributeValue{"Id": {S: aws.String(id)}},
		UpdateExpression: aws.String("SET #status = :Status, StatusTimes.#time = :Time, LastUpdated = :Time"),
		ExpressionAttributeNames: map[string]*string{
			"#status": aws.String("Status"),
			"#time":   aws.String(status),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":Status": {S: aws.String(status)},
			":Time":   {S: aws.String(now3339())},
		},
		ReturnValues: aws.String("ALL_NEW"),
	}

	resp, err := d.client.UpdateItem(input)
	go d.publishStateDuration(resp, status)

	return err
}

func (d *ddb) SetFailed(id, reason string) error {
	status := types.StatusFailed
	input := &dynamodb.UpdateItemInput{
		TableName:        aws.String(d.table),
		Key:              map[string]*dynamodb.AttributeValue{"Id": {S: aws.String(id)}},
		UpdateExpression: aws.String("SET #status = :Status, #reason = :Reason, StatusTimes.#time = :Time, LastUpdated = :Time"),
		ExpressionAttributeNames: map[string]*string{
			"#status": aws.String("Status"),
			"#reason": aws.String("FailureReason"),
			"#time":   aws.String(status),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":Status": {S: aws.String(status)},
			":Reason": {S: aws.String(reason)},
			":Time":   {S: aws.String(now3339())},
		},
		ReturnValues: aws.String("ALL_NEW"),
	}

	resp, err := d.client.UpdateItem(input)
	go d.publishStateDuration(resp, status)

	return err
}

func (d *ddb) publishStateDuration(out *dynamodb.UpdateItemOutput, status string) {
	upload := &types.Upload{}
	if err := dynamodbattribute.UnmarshalMap(out.Attributes, upload); err != nil {
		return
	}

	var previousState string

	switch status {
	case types.StatusPendingTranscode:
		previousState = types.StatusCreated
	case types.StatusTranscoding:
		previousState = types.StatusPendingTranscode
	case types.StatusProcessed:
		previousState = types.StatusTranscoding
	}

	startTime := upload.StatusTimes[previousState]
	endTime := upload.StatusTimes[status]

	duration := endTime.Sub(startTime)
	d.logger.Infof("Job state duration: %v seconds", duration.Seconds())

	m := cw.Metric{
		Name:  "StateDuration",
		Value: duration.Seconds(),
		Unit:  cw.UNIT_Seconds,
	}
	m.AddDimension("State", previousState)
	d.cloudwatch.PutMetric(&m)
}
