package integration

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"math"
	"net/http"
	"os"
	"strings"
	"time"

	"code.justin.tv/vodsvc/aws/awsconfig"
	"code.justin.tv/vodsvc/aws/cloudformation"
	"code.justin.tv/vodsvc/aws/s3"
	"code.justin.tv/vodsvc/httpserver"
	"code.justin.tv/vodsvc/vhs/src/api"
	"code.justin.tv/vodsvc/vhs/src/dynamodb"
	"code.justin.tv/vodsvc/vhs/src/types"
	log "go.uber.org/zap"
)

const (
	testVideo     = "test.mp4"
	testVideoPath = "/tmp/" + testVideo
)

var (
	Host        string
	logger      *log.SugaredLogger
	uploadTable dynamodb.UploadTable
)

func Run() error {
	InitEnvironment()

	if err := PrepareTestArtifacts(); err != nil {
		return err
	}

	_, err := Create()
	if err != nil {
		return err
	}

	/*
		uploadId := u.Upload.Id

			if err = Upload(uploadId, uploadToken); err != nil {
				return err
			}

			if err = Complete(uploadId, uploadToken); err != nil {
				return err
			}

			if err := WaitStatus(uploadId); err != nil {
				return err
			}
	*/

	logger.Info("Integration test success")
	return nil
}

func InitEnvironment() {
	logger = httpserver.DefaultLogger().Sugar()
	awsconfig.SetLogger(logger.Debug)

	cf := cloudformation.NewFromEnv()
	tableName := cf.MustGetPhysicalResourceId("UploadTable")
	uploadTable = dynamodb.New(tableName, logger)
	if Host == "" {
		Host = fmt.Sprintf("http://%v.%v.elasticbeanstalk.com",
			os.Getenv("ENV_NAME"), "bogus")
	}
	logger.Info("Running integration test against endpoint: ", Host)
}

func PrepareTestArtifacts() error {
	cf := cloudformation.New("vhs")
	s3bucket := cf.MustGetPhysicalResourceId("S3TestArtifacts")
	s3client := s3.New(s3bucket)

	return s3client.CacheObject(testVideo, testVideoPath)
}

func WaitStatus(uploadId string) error {
	maxWait := 90 * time.Second
	sleepTime := 10 * time.Second
	attempts := int(maxWait / sleepTime)

	var status string
	status = types.StatusPendingTranscode

	logger.Info("Waiting for video status: " + status)
	for i := 0; i < attempts; i++ {
		upload, err := uploadTable.GetUpload(uploadId)
		if err != nil {
			return err
		}

		logger.Info("Video status: " + upload.Status)
		if upload.Status != status {
			time.Sleep(30 * time.Second)
		} else {
			return nil
		}
	}

	return errors.New("Upload did not change its status in time")
}

func Create() (*api.CreateUploadResponse, error) {
	rsp, err := request(http.MethodPost, "/uploads", []byte{})
	if err != nil {
		return nil, err
	}

	output := &api.CreateUploadResponse{}
	err = json.NewDecoder(strings.NewReader(rsp)).Decode(output)

	return output, err
}

func Upload(uploadId, uploadToken string) error {
	file, err := os.Open(testVideoPath)
	if err != nil {
		return err
	}

	fileInfo, err := file.Stat()
	if err != nil {
		return err
	}

	partSize := 5 * 1024 * 1024
	fileSize := float64(fileInfo.Size())
	parts := int(math.Ceil(fileSize / float64(partSize)))

	data := make([]byte, partSize)

	for i := 1; i <= parts; i++ {
		path := fmt.Sprintf("/uploads/%v?part=%d", uploadId, i)
		_, err := file.Read(data)
		if err != nil {
			return err
		}

		if _, err := request(http.MethodPost, path, data, authHeader(uploadToken)); err != nil {
			return err
		}
	}

	return nil
}

func Complete(uploadId, uploadToken string) error {
	path := fmt.Sprintf("/uploads/%v/complete", uploadId)

	_, err := request(http.MethodPost, path, []byte{}, authHeader(uploadToken))

	return err
}

func request(method, path string, body []byte, headers ...map[string]string) (string, error) {
	url := Host + path
	request, err := http.NewRequest(method, url, bytes.NewBuffer(body))
	if err != nil {
		return "", err
	}

	if len(headers) > 0 {
		for h, v := range headers[0] {
			request.Header.Set(h, v)
		}
	}

	httpClient := http.Client{}
	startTime := time.Now()
	rsp, err := httpClient.Do(request)
	if err != nil {
		return "", err
	}
	endTime := time.Now()

	out, err := ioutil.ReadAll(rsp.Body)
	if err != nil {
		return "", err
	}
	defer rsp.Body.Close()

	logger.Infof("%v %v, status=%v, duration=%v", method, url, rsp.StatusCode, endTime.Sub(startTime))

	if rsp.StatusCode >= 300 {
		return "", errors.New(fmt.Sprint(rsp.StatusCode, ": ", string(out)))
	}

	return string(out), nil
}

func authHeader(uploadToken string) map[string]string {
	return map[string]string{
		"Authorization": "Bearer " + uploadToken,
	}
}
