package lambda

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"os"
	"strings"

	"code.justin.tv/vodsvc/httpserver"

	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
)

type LambdaFunc func(context.Context, events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error)

// The API signature that methods should implement.
type APIFunc func(context.Context, events.APIGatewayProxyRequest) (interface{}, *httpserver.HTTPError)

// Only invokes Lambda path when run inside a Lambda function, otherwise just
// execute it locally and use shared AWS credentials file.
func EnvStart(f APIFunc) {
	if len(os.Getenv("LAMBDA_TASK_ROOT")) > 0 {
		lambda.Start(wrap(f))
	} else {
		runLocal(f)
	}
}

func runLocal(f APIFunc) {
	logger := httpserver.DefaultLogger().Sugar()
	params := map[string]string{}

	if len(os.Args) > 1 {
		for _, a := range os.Args[1:] {
			kv := strings.Split(a, "=")
			k := kv[0]
			v := kv[1]
			params[k] = v
		}
	}

	req := events.APIGatewayProxyRequest{
		PathParameters:        params,
		QueryStringParameters: params,
	}
	rsp, err := f(context.TODO(), req)
	if err != nil {
		logger.Error(err.Err + ": " + err.Message)
		os.Exit(1)
	}

	jsonString, jerr := json.MarshalIndent(rsp, "", "  ")
	if jerr != nil {
		logger.Error(jerr)
		os.Exit(1)
	}

	fmt.Println(string(jsonString))
}

func wrap(f APIFunc) LambdaFunc {
	return func(c context.Context, r events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error) {
		code := 200
		obj, httperr := f(c, r)
		if httperr != nil {
			code = httperr.Code
			obj = httperr
		}

		jsonString, err := json.Marshal(obj)
		if err != nil {
			code := http.StatusInternalServerError
			return response(http.StatusText(code), code), nil
		}

		return response(string(jsonString), code), nil
	}
}

func response(body string, code int) events.APIGatewayProxyResponse {
	return events.APIGatewayProxyResponse{Body: body, StatusCode: code}
}
