package main

import (
	"context"
	"fmt"
	"os"
	"strings"
	"time"

	"code.justin.tv/vodsvc/aws/s3"
	"code.justin.tv/vodsvc/vhs/src/s3helper"
	"code.justin.tv/vodsvc/vhs/src/types"
	"code.justin.tv/vodsvc/vhs/src/vhsclient"
	"code.justin.tv/vodsvc/vhs/src/vhslog"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"go.uber.org/zap"
)

var (
	logger       *zap.SugaredLogger = vhslog.Logger
	videoBucket  s3.S3              = s3helper.NewVideoBucket()
	uploadBucket s3.S3              = s3helper.NewUploadBucket()
	vhsClient    *vhsclient.Client  = vhsclient.New("")
)

const (
	testBucket = "vhs.us-west-2.test-artifacts"
	testObject = "test.mp4"
)

func main() {
	vhsClient.SetLogger(logger.Debug)
	if len(os.Getenv("LAMBDA_TASK_ROOT")) > 0 {
		lambda.Start(TestUpload)
	} else {
		TestUpload(context.TODO(), events.CloudWatchEvent{})
	}
}

func TestUpload(ctx context.Context, e events.CloudWatchEvent) {
	upload, err := vhsClient.CreateUpload()
	if err != nil {
		logger.Fatal(err)
	}

	targetKey := strings.TrimPrefix(upload.S3URI, "s3://"+upload.Upload.S3Bucket+"/")

	if err := uploadBucket.CopyObject(testBucket, testObject, targetKey); err != nil {
		logger.Fatal(err)
	}

	if err := waitStatus(upload.Upload, types.StatusProcessed, 120); err != nil {
		logger.Fatal(err)
	}

	if _, err := vhsClient.GetVideo(upload.Upload.Id); err != nil {
		logger.Fatal(err)
	}

	if _, err := vhsClient.ListVideos(); err != nil {
		logger.Fatal(err)
	}

	if _, err := vhsClient.GetManifest(upload.Upload.Id); err != nil {
		logger.Fatal(err)
	}

	if err := vhsClient.DeleteVideo(upload.Upload.Id); err != nil {
		logger.Fatal(err)
	}
}

func waitStatus(upload types.Upload, status string, timeout int) error {
	return retry(timeout, func() error {
		u, err := vhsClient.GetUpload(upload.Id)
		if err != nil {
			return err
		}
		if u.Status == status {
			return nil
		}
		return fmt.Errorf("Upload didn't enter expected status; id=%v, status=%v", u.Id, u.Status)
	})
}

func retry(timeout int, f func() error) error {
	var err error
	interval := 5
	attempts := timeout / interval
	for i := 0; i < attempts; i++ {
		err = f()
		if err == nil {
			return nil
		}
		time.Sleep(time.Duration(interval) * time.Second)
	}
	return err
}
