package main

import (
	"context"
	"encoding/json"
	"os"
	"path/filepath"
	"time"

	"code.justin.tv/vodsvc/aws/batch"
	"code.justin.tv/vodsvc/aws/s3"
	sts "code.justin.tv/vodsvc/sts/src/transcoder"
	"code.justin.tv/vodsvc/vhs/src/dynamodb"
	"code.justin.tv/vodsvc/vhs/src/lambda"
	"code.justin.tv/vodsvc/vhs/src/s3helper"
	"code.justin.tv/vodsvc/vhs/src/transcode"
	"code.justin.tv/vodsvc/vhs/src/types"
	"code.justin.tv/vodsvc/vhs/src/vhslog"
	"go.uber.org/zap"
)

var (
	logger      *zap.SugaredLogger   = vhslog.Logger
	uploadTable dynamodb.UploadTable = dynamodb.NewUploadTable()
	videoTable  dynamodb.VideoTable  = dynamodb.NewVideoTable()
	videoBucket s3.S3                = s3helper.NewVideoBucket()
	transcoder  transcode.Transcoder = transcode.NewFromEnv()
)

func main() {
	lambda.StartBatchEvent(CompleteTranscode)
}

func CompleteTranscode(ctx context.Context, msg batch.BatchMessage) {
	uploadId, ok := msg.Parameters["UploadId"]
	if !ok {
		logger.Fatalw("Could not retrieve UploadId from parameters", "msg", msg)
	}

	upload, err := uploadTable.GetUpload(ctx, uploadId)
	if err != nil {
		logger.Fatalw("Could not retrieve Upload from DynamoDB", "id", uploadId, "err", err)
	}

	if upload.Id == "" {
		logger.Fatalw("UploadId not found", "id", uploadId)
	}

	switch msg.Status {
	case "RUNNING":
		setTranscoding(ctx, upload.Id)
	case "FAILED":
		failUpload(ctx, upload.Id, "Batch job failed")
	case "SUCCEEDED":
		manifestBytes, err := videoBucket.GetObject(filepath.Join(upload.S3Prefix, "video.json"))
		if err != nil {
			failUpload(ctx, uploadId, "Error retrieving video manifest from S3: "+err.Error())
		}

		var manifest sts.Video
		if err := json.Unmarshal(manifestBytes, &manifest); err != nil {
			failUpload(ctx, uploadId, "Error unmarshalling manifest to JSON: "+err.Error())
		}

		switch manifest.Status {
		case sts.StatusSuccess:
			handleSuccess(ctx, manifest, upload)
		case sts.StatusFailure:
			failUpload(ctx, upload.Id, manifest.FailureReason, types.ErrCode(manifest.FailureCode))
		default:
			logger.Fatalw("Unknown transcoder status: " + manifest.Status)
		}
	default:
		logger.Fatalw("Unknown Batch job status: " + msg.Status)
	}
}

func setTranscoding(ctx context.Context, id string) {
	status := types.StatusTranscoding
	err := uploadTable.SetStatus(ctx, id, status)
	if err != nil {
		failUpload(ctx, id, "Error setting upload status: "+err.Error())
	}
	logger.Info("Set status to '" + status + "' in DDB")
}

func handleSuccess(ctx context.Context, manifest sts.Video, u types.Upload) {
	video := types.Video{
		Id:         u.Id,
		CreateTime: time.Now().UTC(),
		S3Bucket:   u.VideoBucket,
		S3Prefix:   u.S3Prefix,
		VideoType:  types.TypeUpload,
	}

	video = transcoder.ParseManifest(video, manifest)

	if err := videoTable.CreateVideo(ctx, video); err != nil {
		failUpload(ctx, u.Id, "Error creating video in Dynamo table: "+err.Error())
	}

	if err := uploadTable.SetStatus(ctx, u.Id, types.StatusProcessed); err != nil {
		failUpload(ctx, u.Id, "Error setting upload status: "+err.Error())
	}
}

func failUpload(ctx context.Context, id, reason string, errCodes ...types.ErrCode) {
	code := types.ErrInternalFailure
	if len(errCodes) > 0 {
		code = errCodes[0]
	}

	err := uploadTable.SetFailed(ctx, id, code)
	if err != nil {
		logger.Fatalw("Error setting failed status in DynamoDB", "id", id, "err", err)
	}

	logFunc := logger.Warnw
	if code == types.ErrInternalFailure {
		logFunc = logger.Fatalw
	}

	logFunc("Failing upload", "id", id, "code", code, "reason", reason)
	os.Exit(0)
}
