package main

import (
	"context"

	"code.justin.tv/vodsvc/vhs/src/dynamodb"
	"code.justin.tv/vodsvc/vhs/src/transcode"
	"code.justin.tv/vodsvc/vhs/src/types"
	"code.justin.tv/vodsvc/vhs/src/vhslog"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"go.uber.org/zap"
)

var (
	logger      *zap.SugaredLogger   = vhslog.Logger
	uploadTable dynamodb.UploadTable = dynamodb.NewUploadTable()
	transcoder  transcode.Transcoder = transcode.NewFromEnv()
)

func main() {
	lambda.Start(CompleteUpload)
}

func CompleteUpload(ctx context.Context, e events.S3Event) {
	for _, r := range e.Records {
		uploadId := r.S3.Object.Key[0:36]

		u, err := uploadTable.GetUpload(ctx, uploadId)
		if err != nil {
			logger.Fatalw("Cannot retrieve upload from DynamoDB", "id", uploadId, "err", err)
		}
		if u.Id == "" {
			logger.Fatalw("UploadId not found", "id", uploadId)
		}

		if u.Status != types.StatusCreated {
			logger.Warnw("Upload has already started processing", "id", uploadId)
			return
		}

		if err := transcoder.StartTranscode(uploadId, uploadTable.TableName()); err != nil {
			logger.Errorw("Error starting transcode job", "id", uploadId, "err", err)
			if err := uploadTable.SetFailed(ctx, uploadId, types.ErrInternalFailure); err != nil {
				logger.Fatalw("Error setting upload status", "id", uploadId, "err", err)
			}
		}

		if err := uploadTable.SetStatus(ctx, uploadId, types.StatusPendingTranscode); err != nil {
			logger.Fatalw("Error setting upload status", "id", uploadId, "err", err)
		}
		if err := uploadTable.SetSize(ctx, uploadId, r.S3.Object.Size); err != nil {
			logger.Fatalw("Error setting upload size", "id", uploadId, "err", err)
		}
	}
}
