package main

import (
	"context"
	"encoding/json"
	"fmt"
	"net/url"
	"os"
	"time"

	"code.justin.tv/vodsvc/aws/awsconfig"
	"code.justin.tv/vodsvc/httpserver"
	"code.justin.tv/vodsvc/vhs/src/internal/awsutils"
	"code.justin.tv/vodsvc/vhs/src/lambda"
	"code.justin.tv/vodsvc/vhs/src/types"
	"code.justin.tv/vodsvc/vhs/src/vhslog"

	"code.justin.tv/video/lvsapi/streamkey"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/golang/protobuf/ptypes"
	"go.uber.org/zap"
)

var (
	logger *zap.SugaredLogger = vhslog.Logger
)

const (
	// Return this value if a specific streamkey metadata key is not specified
	defaultStreamKeyParameterValue = "Not Specified"
	defaultStreamkeyExpiry         = 300
	rtmpServerURL                  = "rtmp://rtmplive.twitch.tv/app"
	playBackURLTemplate            = "https://usher.ttvnw.net/api/lvs/hls/%s.m3u8?allow_source=true&player_backend=mediaplayer"
)

func main() {
	lambda.StartAPIGateway(CreateStreamKey)
}

// CreateStreamKey will return a valid streamkey along with a URL to playback video.
func CreateStreamKey(ctx context.Context, r events.APIGatewayProxyRequest) (interface{}, *httpserver.HTTPError) {
	accountID, err := lambda.GetAccountID(r)
	if err != nil {
		// TODO: Change this to the correct status code
		return nil, httpserver.BadRequest("failed to authorize request")
	}
	privData := streamkey.PrivateData{
		CustomerId: accountID,
	}

	var req types.CreateStreamKeyRequest
	err = json.Unmarshal([]byte(r.Body), &req)
	if err != nil {
		return nil, httpserver.BadRequest(fmt.Sprintf("error parsing request: %s", err))
	}

	if req.ContentID == "" {
		return nil, httpserver.BadRequest("'contentID' is a required parameter.")
	}
	privData.ContentId = req.ContentID
	if !lambda.IsValidContentID(privData.ContentId) {
		return nil, httpserver.BadRequest("'contentID' is invalid")
	}

	if req.TTLSeconds == 0 {
		req.TTLSeconds = defaultStreamkeyExpiry
	}

	expiration := time.Now().Add(time.Duration(req.TTLSeconds) * time.Second)
	privData.ExpirationTime, err = ptypes.TimestampProto(expiration)
	if err != nil {
		return nil, httpserver.BadRequest(fmt.Sprintf("invalid ttlSeconds"))
	}

	// CR: Consider moving SNS config to seperate endpoint
	if req.SnsNotificationEndpoint != "" {
		if err := awsutils.ValidateSnsArn(req.SnsNotificationEndpoint); err != nil {
			return nil, httpserver.BadRequest(fmt.Sprintf("invalid snsNotificationEndpoint: %s", err.Error()))
		}
		privData.SnsNotificationEndpoint = req.SnsNotificationEndpoint
	}

	if req.CdnURL != "" {
		if _, err := url.ParseRequestURI(req.CdnURL); err != nil {
			return nil, httpserver.BadRequest(fmt.Sprintf("invalid cdnURL: %s", err.Error()))
		}
		privData.CdnUrl = req.CdnURL
	}

	logger.Debug(ctx, fmt.Sprintf("CustomerID: %s, ContentID: %s", privData.CustomerId, privData.ContentId))

	privData.ChannelName = fmt.Sprintf("lvs.%s.%s", privData.CustomerId, privData.ContentId)
	privData.LowLatencyMode = false

	streamKey := streamkey.NewV1(privData.CustomerId, &privData)
	secretSource := newSecretSource()
	encryptedStreamKey, err := streamkey.Encrypt(ctx, secretSource, streamKey)
	if err != nil {
		logger.Errorf("unable to encrypt stream key: %s", err)
		return nil, httpserver.InternalServerError(err)
	}

	res := types.CreateStreamKeyResponse{
		ContentID:     privData.ContentId,
		CustomerID:    privData.CustomerId,
		PlaybackURL:   fmt.Sprintf(playBackURLTemplate, privData.ChannelName),
		RtmpIngestURL: fmt.Sprintf("%s/%s", rtmpServerURL, encryptedStreamKey),
		RtmpServer:    rtmpServerURL,
		Streamkey:     encryptedStreamKey,
		StreamkeyMetadata: types.StreamkeyMetadataResponse{
			ExpirationTime:          expiration,
			SnsNotificationEndpoint: getStreamKeyParameter(privData.SnsNotificationEndpoint),
			TTLSeconds:              req.TTLSeconds,
			CdnURL:                  getStreamKeyParameter(privData.CdnUrl),
		},
	}

	return res, nil
}

func getStreamKeyParameter(inputParam string) string {
	if inputParam == "" {
		return defaultStreamKeyParameterValue
	}

	return inputParam
}

// TODO: Where should we move this S3 logic / should it be replaced with dynamo calls?
var (
	// SecretS3Bucket is the s3 bucket to use for streamkey secret storage
	SecretS3Bucket = os.Getenv("STREAM_KEY_SECRETS_BUCKET")
	// SecretS3Prefix is the s3 bucket prefix to use when determining streamkey secret key names
	SecretS3Prefix = os.Getenv("STREAM_KEY_SECRETS_BUCKET_PREFIX")
	// SecretS3Duration specifies how long to cache secrets in between requests to s3
	SecretS3Duration = 60 * time.Second
)

func newSecretSource() streamkey.SecretSource {
	var secretSource streamkey.SecretSource

	if SecretS3Bucket == "" {
		// TODO: Allow this state on dev, error out fatally on prod
		logger.Debug("Autogenerating secret as no s3 bucket was specified")
		secret, err := streamkey.GenerateSecret()
		if err != nil {
			logger.Fatal("failed to generate secret: %e", err)
		}

		secretSource = &streamkey.SingleSecretSource{Secret: secret}
	} else {
		secretSource = streamkey.NewS3SecretSource(streamkey.S3SecretSourceConfig{
			S3:           s3.New(awsconfig.Session),
			Bucket:       SecretS3Bucket,
			Prefix:       SecretS3Prefix,
			CacheTimeout: SecretS3Duration,
		})
	}

	return secretSource
}
