package main

import (
	"context"
	"encoding/base64"
	"fmt"
	"os"
	"strings"
	"time"
	"unicode"

	"code.justin.tv/vodsvc/aws/awsconfig"
	"code.justin.tv/vodsvc/httpserver"
	"code.justin.tv/vodsvc/vhs/src/dynamodb"
	"code.justin.tv/vodsvc/vhs/src/lambda"
	"code.justin.tv/vodsvc/vhs/src/s3helper"
	"code.justin.tv/vodsvc/vhs/src/types"
	"code.justin.tv/vodsvc/vhs/src/vhslog"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/kms"
	"github.com/aws/aws-sdk-go/service/sts"
	uuid "github.com/satori/go.uuid"
	"go.uber.org/zap"
)

const defaultUploadDuration = 12 * time.Hour / time.Second
const sourceFile = "source"
const iamPolicy = `
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": ["s3:PutObject","s3:AbortMultipartUpload","s3:ListMultipartUploadParts"],
      "Resource": "arn:aws:s3:::%v/%v"
    }
  ]
}
`

var (
	logger      *zap.SugaredLogger   = vhslog.Logger
	uploadTable dynamodb.UploadTable = dynamodb.NewUploadTable()
)

func main() {
	lambda.StartAPIGateway(CreateUpload)
}

func CreateUpload(ctx context.Context, r events.APIGatewayProxyRequest) (interface{}, *httpserver.HTTPError) {
	var err error

	id := generateVideoId()
	s3prefix := generateS3Prefix(id)
	s3bucket := s3helper.UploadBucket
	videoBucket := s3helper.VideoBucket
	sourceKey := sourceKey(s3prefix)

	upload, err := uploadTable.CreateUpload(ctx, id, s3bucket, s3prefix, videoBucket, sourceFile)
	if err != nil {
		return nil, httpserver.InternalServerError(err)
	}

	creds, err := getCreds(id, s3bucket, sourceKey)
	if err != nil {
		logger.Error("Error getting credentials: " + err.Error())
		return nil, httpserver.InternalServerError(err)
	}

	response := &types.CreateUploadResponse{
		Upload:      upload,
		Credentials: creds,
		S3URI:       s3URI(s3bucket, sourceKey),
	}

	return response, nil
}

func getCreds(id, s3bucket, sourceKey string) (types.Credentials, error) {
	policy := squish(fmt.Sprintf(iamPolicy, s3bucket, sourceKey))

	accessKey := os.Getenv("FEDERATION_ACCESS")
	secretKey := os.Getenv("FEDERATION_SECRET")
	kmsClient := kms.New(awsconfig.Session)

	b64secret, err := base64.StdEncoding.DecodeString(secretKey)
	if err != nil {
		return types.Credentials{}, fmt.Errorf("Error base64 decoding secret key: %v", err)
	}

	decryptInput := &kms.DecryptInput{
		CiphertextBlob: []byte(b64secret),
	}

	decryptOutput, err := kmsClient.Decrypt(decryptInput)
	if err != nil {
		return types.Credentials{}, fmt.Errorf("Error decrypting secret key: %v", err)
	}

	awsCreds := credentials.NewStaticCredentials(accessKey, string(decryptOutput.Plaintext), "")
	config := aws.Config{Credentials: awsCreds}
	stsClient := sts.New(session.Must(session.NewSession(&config)))

	input := &sts.GetFederationTokenInput{
		DurationSeconds: aws.Int64(int64(defaultUploadDuration)),
		Name:            aws.String(id[0:32]),
		Policy:          aws.String(policy),
	}

	output, err := stsClient.GetFederationToken(input)
	if err != nil {
		return types.Credentials{}, err
	}

	creds := types.Credentials{
		AccessKeyId:     *output.Credentials.AccessKeyId,
		SecretAccessKey: *output.Credentials.SecretAccessKey,
		SessionToken:    *output.Credentials.SessionToken,
		Expiration:      output.Credentials.Expiration.Format(time.RFC3339),
	}

	return creds, nil
}

func squish(str string) string {
	return strings.Map(func(r rune) rune {
		if unicode.IsSpace(r) {
			return -1
		}
		return r
	}, str)
}

func generateVideoId() string {
	return uuid.NewV4().String()
}

func generateS3Prefix(id string) string {
	return id + "/" + uuid.NewV4().String()
}

func sourceKey(s3Prefix string) string {
	return s3Prefix + "/" + sourceFile
}

func s3URI(bucket, key string) string {
	return "s3://" + bucket + "/" + key
}
