package main

import (
	"context"
	"fmt"
	"os"

	"code.justin.tv/vodsvc/aws/s3"
	"code.justin.tv/vodsvc/vhs/src/dynamodb"
	"code.justin.tv/vodsvc/vhs/src/vhslog"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"go.uber.org/zap"
)

var (
	logger     *zap.SugaredLogger  = vhslog.Logger
	videoTable dynamodb.VideoTable = dynamodb.NewVideoTable()
)

const (
	dryRun  = false
	maxKeys = 1000000
)

func main() {
	if len(os.Getenv("LAMBDA_TASK_ROOT")) > 0 {
		lambda.Start(DeleteVideos)
	} else {
		DeleteVideos(context.TODO(), events.CloudWatchEvent{})
	}
}

func DeleteVideos(ctx context.Context, e events.CloudWatchEvent) {
	videos, err := videoTable.GetVideosToDelete(ctx)
	if err != nil {
		logger.Fatal(err)
	}

	for _, v := range videos {
		if dryRun {
			logger.Infow("Deleting video (dry run)", "id", v.Id, "prefix", v.S3Prefix, "bucket", v.S3Bucket)
		} else {
			logger.Infow("Deleting video", "id", v.Id, "prefix", v.S3Prefix, "bucket", v.S3Bucket)
			videoBucket := s3.New(v.S3Bucket)
			err := videoBucket.BatchDeletePrefix(v.S3Prefix, int64(maxKeys))
			if err != nil {
				logger.Fatal(err)
			}

			objs, err := videoBucket.ListObjects(v.S3Prefix)
			if err != nil {
				logger.Fatal(err)
			}
			if len(objs) > 0 {
				logger.Fatal(fmt.Errorf("Files remaining after deletion; bucket=%v, prefix=%v, count=%v", v.S3Bucket, v.S3Prefix, len(objs)))
			}

			if _, err := videoTable.DeleteVideo(ctx, v.Id); err != nil {
				logger.Fatal(err)
			}
		}
	}
}
