package main

import (
	"context"
	"net/http"
	"os"
	"strings"

	"code.justin.tv/vodsvc/vhs/src/dynamodb"
	"code.justin.tv/vodsvc/vhs/src/hls"
	"code.justin.tv/vodsvc/vhs/src/vhslog"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"go.uber.org/zap"
)

var (
	logger     *zap.SugaredLogger  = vhslog.Logger
	videoTable dynamodb.VideoTable = dynamodb.NewVideoTable()
	hlsc       *hls.HLS
)

func main() {
	cloudFrontDomain := os.Getenv("CLOUDFRONT_DOMAIN")
	var err error
	hlsc, err = hls.New(hls.Config{CloudFrontDomain: cloudFrontDomain})
	if err != nil {
		logger.Fatalw("Error creating HLS template: " + err.Error())
	}

	lambda.Start(GetManifest)
}

// GetManifest generate the master manifest for a video
func GetManifest(ctx context.Context, r events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error) {
	id, ok := r.PathParameters["id"]
	if !ok {
		return respondError(http.StatusBadRequest)
	}

	id = strings.TrimSuffix(id, ".m3u8")

	video, err := videoTable.GetVideo(ctx, id)
	if err != nil {
		return respondError(http.StatusInternalServerError)
	}
	if video.Id == "" {
		return respondError(http.StatusNotFound)
	}

	manifest, err := hlsc.WriteMasterManifest(video)
	if err != nil {
		return respondError(http.StatusInternalServerError)
	}

	response := events.APIGatewayProxyResponse{
		StatusCode: http.StatusOK,
		Headers: map[string]string{
			"Content-Type":                "application/vnd.apple.mpegurl",
			"Access-Control-Allow-Origin": "*",
		},
		Body: manifest,
	}
	return response, nil
}

func respondError(code int) (events.APIGatewayProxyResponse, error) {
	return events.APIGatewayProxyResponse{Body: http.StatusText(code), StatusCode: code}, nil
}
