package main

import (
	"context"

	"code.justin.tv/vodsvc/aws/cloudwatch"
	"code.justin.tv/vodsvc/httpserver"
	"code.justin.tv/vodsvc/vhs/src/internal/usher"
	"code.justin.tv/vodsvc/vhs/src/lambda"
	"code.justin.tv/vodsvc/vhs/src/types"
	"code.justin.tv/vodsvc/vhs/src/vhslog"

	"github.com/aws/aws-lambda-go/events"
	"github.com/pkg/errors"
	"go.uber.org/zap"
)

var (
	logger *zap.SugaredLogger = vhslog.Logger
	cw     cloudwatch.CloudWatch
)

func main() {
	lambda.StartAPIGateway(GetStream)
	cw = cloudwatch.New(logger)
}

// GetStream returns information about a stream for a given Content ID
func GetStream(ctx context.Context, r events.APIGatewayProxyRequest) (interface{}, *httpserver.HTTPError) {
	accountID, err := lambda.GetAccountID(r)
	if err != nil {
		return nil, httpserver.BadRequest("failed to retrieve account id")
	}

	contentID, ok := r.PathParameters["content_id"]
	if !ok {
		return nil, httpserver.BadRequest("'contentID' is a required parameter.")
	}
	if !lambda.IsValidContentID(contentID) {
		return nil, httpserver.BadRequest("'contentID' is invalid")
	}

	us, err := usher.NewUsher(cw).GetStreamInfo(accountID, contentID)
	if err == usher.ErrNotFound {
		return nil, httpserver.NotFound("stream is not live")
	} else if err != nil {
		return nil, httpserver.InternalServerError(errors.Wrap(err, "unable to query streams"))
	}

	stream, err := us.Stream(accountID, contentID)
	if err != nil {
		return nil, httpserver.InternalServerError(errors.Wrap(err, "failed to parse usher response"))
	}

	res := types.GetStreamResponse{
		Stream: *stream,
	}

	return res, nil
}
