package main

import (
	"context"

	"code.justin.tv/vodsvc/aws/cloudwatch"
	"code.justin.tv/vodsvc/httpserver"
	"code.justin.tv/vodsvc/vhs/src/internal/constants"
	"code.justin.tv/vodsvc/vhs/src/internal/usher"
	"code.justin.tv/vodsvc/vhs/src/lambda"
	"code.justin.tv/vodsvc/vhs/src/types"
	"code.justin.tv/vodsvc/vhs/src/vhslog"

	"github.com/aws/aws-lambda-go/events"
	"github.com/pkg/errors"
	"go.uber.org/zap"
)

var (
	logger *zap.SugaredLogger = vhslog.Logger
	cw     cloudwatch.CloudWatch
)

func main() {
	lambda.StartAPIGateway(ListStreams)
	cw = cloudwatch.New(logger)
}

// ListStreams returns information about all streams
func ListStreams(ctx context.Context, r events.APIGatewayProxyRequest) (interface{}, *httpserver.HTTPError) {
	accountID, err := lambda.GetAccountID(r)
	if err != nil {
		return nil, httpserver.BadRequest("failed to retrieve account id")
	}

	reqType, ok := r.QueryStringParameters["type"]
	if !ok {
		return nil, httpserver.BadRequest("'type' is a required parameter.")
	}

	if reqType != constants.STREAM_TYPE_ALL && reqType != constants.STREAM_TYPE_PREPARING &&
		reqType != constants.STREAM_TYPE_LIVE {
		return nil, httpserver.BadRequest("accepted values for 'type' should be one of (all, live, or preparing) ")
	}

	list, err := usher.NewUsher(cw).ListStreams(accountID)
	if err != nil {
		return nil, httpserver.InternalServerError(errors.Wrap(err, "unable to query streams"))
	}

	res := types.ListStreamsResponse{
		Streams: []types.Stream{},
	}

	for _, us := range list {
		if !isRequestedStreamType(reqType, us.Status) {
			continue
		}

		s, err := us.Stream(accountID, us.ContentId)
		if err != nil {
			return nil, httpserver.InternalServerError(errors.Wrap(err, "failed to parse usher response"))
		}

		res.Streams = append(res.Streams, *s)
	}

	return res, nil
}

func isRequestedStreamType(reqType string, streamStatus string) bool {
	if reqType == constants.STREAM_TYPE_ALL {
		return true
	}

	if reqType == constants.STREAM_TYPE_LIVE && streamStatus == constants.TRANSCODE_ACTIVE {
		return true
	}

	if reqType == constants.STREAM_TYPE_PREPARING && streamStatus == constants.TRANSCODE_PENDING {
		return true
	}

	return false
}
