package dynamodb

import (
	"context"
	"os"
	"time"

	"github.com/aws/aws-sdk-go/service/dynamodb"
	ddbiface "github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/aws/aws-xray-sdk-go/xray"

	"code.justin.tv/vodsvc/aws/awsconfig"
	"code.justin.tv/vodsvc/aws/cloudwatch"
	"code.justin.tv/vodsvc/vhs/src/types"
	"code.justin.tv/vodsvc/vhs/src/vhslog"
	"go.uber.org/zap"
)

const ScanLimit = 100

type UploadTable interface {
	CreateUpload(ctx context.Context, id, s3Bucket, s3Prefix, videoBucket, sourceFile string) (types.Upload, error)
	GetUpload(ctx context.Context, id string) (types.Upload, error)
	SetStatus(ctx context.Context, id, status string) error
	SetSize(ctx context.Context, id string, size int64) error
	SetFailed(ctx context.Context, id string, reason types.ErrCode) error
	ListUploads(ctx context.Context, startKey string) ([]types.Upload, string, error)
	TableName() string
}

type VideoTable interface {
	CreateVideo(ctx context.Context, video types.Video) error
	GetVideo(ctx context.Context, id string) (types.Video, error)
	DeleteVideo(ctx context.Context, id string) (types.Video, error)
	SetVideoDeleteTime(ctx context.Context, id string, t time.Time) error
	ListVideos(ctx context.Context, startKey string) ([]types.Video, string, error)
	GetVideosToDelete(ctx context.Context) ([]types.Video, error)
	TableName() string
}

func NewVideoTable() VideoTable {
	return New(os.Getenv("VIDEO_TABLE"), vhslog.Logger)
}

func NewUploadTable() UploadTable {
	return New(os.Getenv("UPLOAD_TABLE"), vhslog.Logger)
}

func New(table string, logger *zap.SugaredLogger) *ddb {
	client := dynamodb.New(awsconfig.Session)
	xray.Configure(xray.Config{LogLevel: "trace"})
	xray.AWS(client.Client)
	return &ddb{
		cloudwatch: cloudwatch.New(logger),
		client:     client,
		table:      table,
		logger:     logger,
	}
}

func (d *ddb) TableName() string {
	return d.table
}

type ddb struct {
	cloudwatch cloudwatch.CloudWatch
	client     ddbiface.DynamoDBAPI
	logger     *zap.SugaredLogger
	table      string
}

func now() time.Time {
	return time.Now().UTC()
}

func now3339() string {
	return now().Format(time.RFC3339)
}
