package dynamodb

import (
	"context"
	"time"

	"code.justin.tv/vodsvc/vhs/src/types"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/expression"
)

func (d *ddb) GetVideosToDelete(ctx context.Context) ([]types.Video, error) {
	t := time.Now().UTC()
	videos, err := d.getVideosToDelete(ctx, t, true)
	if err != nil {
		return videos, err
	}

	dayBefore := t.Round(24 * time.Hour).Add(-time.Nanosecond)
	missedVideos, err := d.getVideosToDelete(ctx, dayBefore, false)
	if err != nil {
		return videos, err
	}

	videos = append(videos, missedVideos...)
	return videos, nil
}

func (d *ddb) getVideosToDelete(ctx context.Context, t time.Time, rangeKey bool) ([]types.Video, error) {
	videos := []types.Video{}
	tf := t.Format(TimeFormat)
	df := t.Format(DateFormat)

	keyCondition := expression.Key("DeleteDate").Equal(expression.Value(df))
	if rangeKey {
		rangeCondition := expression.KeyLessThanEqual(expression.Key("DeleteTime"), expression.Value(tf))
		keyCondition = keyCondition.And(rangeCondition)
	}

	expr, err := expression.NewBuilder().WithKeyCondition(keyCondition).Build()
	if err != nil {
		return videos, err
	}

	input := &dynamodb.QueryInput{
		TableName:                 aws.String(d.table),
		Select:                    aws.String("ALL_PROJECTED_ATTRIBUTES"),
		IndexName:                 aws.String("DeletionIndex"),
		KeyConditionExpression:    expr.KeyCondition(),
		ExpressionAttributeNames:  expr.Names(),
		ExpressionAttributeValues: expr.Values(),
	}

	out, err := d.client.QueryWithContext(ctx, input)
	if err != nil {
		return videos, err
	}

	if err := dynamodbattribute.UnmarshalListOfMaps(out.Items, &videos); err != nil {
		return videos, err
	}

	return videos, nil
}
