package hls

import (
	"bytes"
	"errors"
	"html/template"
	"net/url"

	"code.justin.tv/vodsvc/vhs/src/types"
)

var masterManifestTemplate = `#EXTM3U
#EXT-X-TWITCH-INFO:ORIGIN="s3"
{{range $rendition := $.Renditions -}}
#EXT-X-MEDIA:TYPE=VIDEO,GROUP-ID="{{$rendition.Label}}",NAME="{{$rendition.Label}}",AUTOSELECT=YES,DEFAULT=YES
#EXT-X-STREAM-INF:PROGRAM-ID=1,BANDWIDTH={{$rendition.Bitrate}},CODECS="{{$rendition.VideoCodec}},{{$rendition.AudioCodec}}",RESOLUTION="{{$rendition.Width}}x{{$rendition.Height}}",VIDEO="{{$rendition.Label}}"
{{$.CloudFrontDomain}}/{{$.VideoPath}}/{{$rendition.Label}}/{{$rendition.Manifest}}
{{end -}}`

type HLS struct {
	template         *template.Template
	cloudFrontDomain string
}

type Config struct {
	CloudFrontDomain string
}

type templateData struct {
	CloudFrontDomain string
	VideoPath        string
	Renditions       []types.Rendition
}

func New(c Config) (*HLS, error) {
	t, err := template.New("hls_master_manifest").Parse(masterManifestTemplate)
	if err != nil {
		return nil, err
	}
	cloudFrontURL, err := url.Parse(c.CloudFrontDomain)
	if err != nil {
		return nil, errors.New("Unable to parse CloudFront domain: " + err.Error())
	}
	if cloudFrontURL.Scheme == "" {
		cloudFrontURL.Scheme = "https"
	}

	return &HLS{template: t, cloudFrontDomain: cloudFrontURL.String()}, nil
}

func (h *HLS) WriteMasterManifest(video types.Video) (string, error) {
	data := &templateData{
		CloudFrontDomain: h.cloudFrontDomain,
		VideoPath:        video.S3Prefix,
		Renditions:       video.Renditions,
	}
	writer := bytes.NewBufferString("")
	err := h.template.Execute(writer, data)
	if err != nil {
		return "", err
	}
	return writer.String(), nil
}
