package usher

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"code.justin.tv/vodsvc/aws/cloudwatch"
	"code.justin.tv/vodsvc/vhs/src/internal/constants"
	"code.justin.tv/vodsvc/vhs/src/internal/httputil"
	"code.justin.tv/vodsvc/vhs/src/types"
)

const requestTimeout = 3 * time.Second

// ErrNotFound indicates a stream is not live
var ErrNotFound = errors.New("stream not found")

// Usher is just a struct which ties up the API's and make testing easier
type Usher struct {
	client  httputil.HttpClient
	statter cloudwatch.CloudWatch
}

// NewUsher function returns instance of Usher with the client set
func NewUsher(cw cloudwatch.CloudWatch) *Usher {
	usher := Usher{
		client: &http.Client{
			Timeout: requestTimeout,
		},
		statter: cw,
	}
	return &usher
}

// UsherStreamResponse holds the basic usher data to populate an LVS response
type UsherStreamResponse struct {
	Channel      string `json:"channel"`
	StartedOn    int64  `json:"started_on"`
	ChannelCount int    `json:"channel_count"`
	LVSMetadata  string `json:"lvs_metadata"`
	Status       string `json:"status"`
	CustomerId   string `json:"customer_id"`
	ContentId    string `json:"content_id"`
	StreamId     uint64 `json:"destination"`
}

// Stream converts an UsherStreamResponse to a Stream
func (u UsherStreamResponse) Stream(CustomerID, ContentID string) (*types.Stream, error) {
	startTime := time.Unix(u.StartedOn, 0)

	var streamStatus string
	if u.Status == constants.TRANSCODE_ACTIVE {
		streamStatus = constants.STATUS_LIVE
	} else {
		streamStatus = constants.STATUS_PREPARING
	}

	// TODO - We need to plug in starvation data into usher and use that for making decisions if stream is healthy
	res := types.Stream{
		CustomerID: CustomerID,
		ContentID:  ContentID,
		PlaybackURL: fmt.Sprintf("https://usher.ttvnw.net/api/lvs/hls/%s.m3u8?allow_source=true&player_backend=mediaplayer",
			getChannelNameLvsStreams(CustomerID, ContentID)),
		StartTime:       startTime,
		DurationSeconds: int64(time.Since(startTime).Seconds()),
		Status:          streamStatus,
		HealthStatus:    constants.HEALTH_STATUS_STABLE,
		HealthReason:    fmt.Sprintf("Healthy stream"),
	}

	return &res, nil
}

func getChannelNameLvsStreams(customerID string, contentID string) string {
	return fmt.Sprintf("%s.%s.%s", "lvs", customerID, contentID)
}

// ListStreams will return a list of usher streams for given customerId
func (u *Usher) ListStreams(customerId string) ([]UsherStreamResponse, error) {
	url := fmt.Sprintf("%s/%s.json", "http://usher.justin.tv/hls_transcode/lvs_customer_transcodes", customerId)

	// Track latency
	start := time.Now()
	rsp, err := u.client.Get(url)
	if u.statter != nil {
		m := cloudwatch.Metric{
			Name:  "Usher.ListStreams",
			Value: float64(time.Since(start).Nanoseconds() / 1000000),
			Unit:  cloudwatch.UNIT_Milliseconds,
		}
		u.statter.PutMetric(&m)
	}

	if err != nil {
		return nil, err
	}
	defer func() {
		if err := rsp.Body.Close(); err != nil {
			fmt.Printf("Unable to close response body %s", err)
		}
	}()

	lst := []UsherStreamResponse{}

	if rsp.StatusCode == http.StatusNotFound {
		// No streams live
		return lst, nil
	} else if rsp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("returned error code : %d ", rsp.StatusCode)
	}

	list, err := ioutil.ReadAll(rsp.Body)
	if err != nil {
		return nil, err
	}

	// don't try to unmarshal usher's non compliant empty response
	if len(list) < 3 {
		return lst, nil
	}

	if err := json.Unmarshal(list, &lst); err != nil {
		return nil, err
	}

	return lst, nil
}

// GetStreamInfo return the information around a given usher stream in its streams table
func (u *Usher) GetStreamInfo(customerId string, contentId string) (*UsherStreamResponse, error) {
	url := fmt.Sprintf("%s/%s/%s.json", "http://usher.justin.tv/hls_transcode/lvs_channel", customerId, contentId)
	// Track latency
	start := time.Now()
	rsp, err := u.client.Get(url)
	if u.statter != nil {
		m := cloudwatch.Metric{
			Name:  "Usher.GetStreamInfo",
			Value: float64(time.Since(start).Nanoseconds() / 1000000),
			Unit:  cloudwatch.UNIT_Milliseconds,
		}
		u.statter.PutMetric(&m)
	}

	if err != nil {
		return nil, err
	}
	defer func() {
		if err := rsp.Body.Close(); err != nil {
			fmt.Printf("Unable to close response body %s", err)
		}
	}()

	if rsp.StatusCode == http.StatusNotFound {
		return nil, ErrNotFound
	} else if rsp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("returned error code : %d ", rsp.StatusCode)
	}

	st, err := ioutil.ReadAll(rsp.Body)
	if err != nil {
		return nil, err
	}

	var str []UsherStreamResponse

	// don't try to unmarshal usher's non compliant empty response
	if len(st) < 3 {
		return nil, errors.New("Stream Not found")
	}

	if err := json.Unmarshal(st, &str); err != nil {
		return nil, err
	}

	return &str[0], nil
}
