package usher

import (
	"net/http"
	"testing"

	"fmt"

	"bytes"
	"io/ioutil"

	"strings"

	"code.justin.tv/vodsvc/vhs/src/internal/httputil/httputilfakes"
	"github.com/stretchr/testify/require"
)

// Test the case when usher returns an error
func TestGetStreamInfoUsherError(t *testing.T) {
	mockHttpClient := httputilfakes.FakeHttpClient{}
	mockHttpClient.GetReturns(nil, fmt.Errorf("random error no idea"))
	usher := Usher{
		client: &mockHttpClient,
	}
	response, err := usher.GetStreamInfo("test_customer", "test_content_id")
	require.Nil(t, response)
	require.NotNil(t, err)
	require.Contains(t, err.Error(), "random error no idea")
}

func TestGetStreamNonSuccessfulResponseCode(t *testing.T) {
	mockHttpClient := httputilfakes.FakeHttpClient{}
	body := `[{"url":"hls_transcode/lvs/amzn_live/test.json" , "type":"error" , "error":"Unauthorized"}]`
	resp := &http.Response{
		Status:        "Unauthorized",
		StatusCode:    http.StatusUnauthorized,
		Proto:         "HTTP/1.1",
		ProtoMajor:    1,
		ProtoMinor:    1,
		Body:          ioutil.NopCloser(bytes.NewBufferString(body)),
		ContentLength: int64(len(body)),
		Request:       &http.Request{},
		Header:        make(http.Header, 0),
	}
	mockHttpClient.GetReturns(resp, nil)
	usher := Usher{
		client: &mockHttpClient,
	}
	response, err := usher.GetStreamInfo("test_customer", "test_content_id")
	require.Nil(t, response)
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "401"))
}

func TestGetStreamSuccess(t *testing.T) {
	mockHttpClient := httputilfakes.FakeHttpClient{}
	body, err := ioutil.ReadFile("testdata/get_stream_response.txt")
	require.Nil(t, err)
	resp := &http.Response{
		Status:        "OK",
		StatusCode:    http.StatusOK,
		Proto:         "HTTP/1.1",
		ProtoMajor:    1,
		ProtoMinor:    1,
		Body:          ioutil.NopCloser(bytes.NewBufferString(string(body))),
		ContentLength: int64(len(body)),
		Request:       &http.Request{},
		Header:        make(http.Header, 0),
	}
	mockHttpClient.GetReturns(resp, nil)
	usher := Usher{
		client: &mockHttpClient,
	}
	response, err := usher.GetStreamInfo("amzn_live", "test")
	require.NotNil(t, response)
	require.Nil(t, err)
	require.Equal(t, "amzn_live", response.CustomerId)
	require.Equal(t, "test", response.ContentId)
	require.Equal(t, "active", response.Status)
	require.Equal(t, int64(0), response.StartedOn)

}

func TestGetStreamUsher404(t *testing.T) {
	mockHttpClient := httputilfakes.FakeHttpClient{}
	body := `[{"url":"hls_transcode/lvs/amzn_live/test.json" , "type":"error" , "error":"Unauthorized"}]`
	resp := &http.Response{
		Status:        "Not Found",
		StatusCode:    http.StatusNotFound,
		Proto:         "HTTP/1.1",
		ProtoMajor:    1,
		ProtoMinor:    1,
		Body:          ioutil.NopCloser(bytes.NewBufferString(body)),
		ContentLength: int64(len(body)),
		Request:       &http.Request{},
		Header:        make(http.Header, 0),
	}
	mockHttpClient.GetReturns(resp, nil)
	usher := Usher{
		client: &mockHttpClient,
	}
	response, err := usher.GetStreamInfo("test_customer", "test_content_id")
	require.Nil(t, response)
	require.NotNil(t, err)
	require.Equal(t, err, ErrNotFound)
}

func TestUsherListStreamsError(t *testing.T) {
	mockHttpClient := httputilfakes.FakeHttpClient{}
	mockHttpClient.GetReturns(nil, fmt.Errorf("random error no idea"))
	usher := Usher{
		client: &mockHttpClient,
	}
	response, err := usher.ListStreams("test_customer")
	require.Nil(t, response)
	require.NotNil(t, err)
	require.Contains(t, err.Error(), "random error no idea")
}

func TestUsherListStreamsNonSuccessfulResponseCode(t *testing.T) {
	mockHttpClient := httputilfakes.FakeHttpClient{}
	body := `[{"url":"hls_transcode/lvs/amzn_live/test.json" , "type":"error" , "error":"Unauthorized"}]`
	usher := Usher{
		client: &mockHttpClient,
	}
	resp := &http.Response{
		Status:        "Unauthorized",
		StatusCode:    http.StatusUnauthorized,
		Proto:         "HTTP/1.1",
		ProtoMajor:    1,
		ProtoMinor:    1,
		Body:          ioutil.NopCloser(bytes.NewBufferString(body)),
		ContentLength: int64(len(body)),
		Request:       &http.Request{},
		Header:        make(http.Header, 0),
	}
	mockHttpClient.GetReturns(resp, nil)

	response, err := usher.ListStreams("amzn_live")
	require.Nil(t, response)
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "401"))
}

func TestUsherListStreamsSuccessfulResponse(t *testing.T) {
	mockHttpClient := httputilfakes.FakeHttpClient{}
	body := `[{"customer_id":"amzn_live" , "content_id":"test" , "status":"active"},{"customer_id":"amzn_live" , "content_id":"test2" , "status":"pending"}]`
	usher := Usher{
		client: &mockHttpClient,
	}
	resp := &http.Response{
		Status:        "OK",
		StatusCode:    http.StatusOK,
		Proto:         "HTTP/1.1",
		ProtoMajor:    1,
		ProtoMinor:    1,
		Body:          ioutil.NopCloser(bytes.NewBufferString(body)),
		ContentLength: int64(len(body)),
		Request:       &http.Request{},
		Header:        make(http.Header, 0),
	}
	mockHttpClient.GetReturns(resp, nil)

	response, err := usher.ListStreams("amzn_live")
	require.NotNil(t, response)
	require.Nil(t, err)
	require.Equal(t, 2, len(response))
}

func TestUsherListStreamsUsher404(t *testing.T) {
	mockHttpClient := httputilfakes.FakeHttpClient{}
	body := `[{"url":"hls_transcode/lvs/amzn_live/test.json" , "type":"error" , "error":"Unauthorized"}]`
	usher := Usher{
		client: &mockHttpClient,
	}
	resp := &http.Response{
		Status:        "Not Found",
		StatusCode:    http.StatusNotFound,
		Proto:         "HTTP/1.1",
		ProtoMajor:    1,
		ProtoMinor:    1,
		Body:          ioutil.NopCloser(bytes.NewBufferString(body)),
		ContentLength: int64(len(body)),
		Request:       &http.Request{},
		Header:        make(http.Header, 0),
	}
	mockHttpClient.GetReturns(resp, nil)
	response, err := usher.ListStreams("amzn_live")
	require.NotNil(t, response)
	require.Nil(t, err)
	require.Len(t, response, 0)
}
