package lambda

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"os"

	"code.justin.tv/vodsvc/httpserver"
	"code.justin.tv/vodsvc/vhs/src/vhslog"
	"go.uber.org/zap"

	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
)

var (
	logger *zap.SugaredLogger = vhslog.Logger
)

// The API signature the aws-lambda-go package expects
type APIGatewayFunc func(context.Context, events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error)

// The API signature that methods should implement.
type APIFunc func(context.Context, events.APIGatewayProxyRequest) (interface{}, *httpserver.HTTPError)

// Only invokes Lambda path when run inside a Lambda function, otherwise just
// execute it locally.
func StartAPIGateway(f APIFunc) {
	if inLambda() {
		lambda.Start(wrapAPIGateway(f))
	} else {
		apiGatewayLocal(f)
	}
}

func apiGatewayLocal(f APIFunc) {
	logger := vhslog.Logger
	params := parseArgs()
	body, _ := json.Marshal(params)

	req := events.APIGatewayProxyRequest{
		PathParameters:        params,
		QueryStringParameters: params,
		Body: string(body),
	}
	req.RequestContext.Identity.AccountID = params["accountID"]

	rsp, err := f(context.TODO(), req)
	if err != nil {
		logger.Error(err.Err + ": " + err.Message)
		logger.Error(err.Cause)
		os.Exit(1)
	}

	jsonString, jerr := json.MarshalIndent(rsp, "", "  ")
	if jerr != nil {
		logger.Error(jerr)
		os.Exit(1)
	}

	fmt.Println(string(jsonString))
}

// A convenience layer that executes common code snippets for lambda functions
// that implement API gateway request/responses.
func wrapAPIGateway(f APIFunc) APIGatewayFunc {
	return func(c context.Context, r events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error) {
		code := 200
		obj, httperr := f(c, r)
		if httperr != nil {
			code = httperr.Code
			obj = httperr
		}

		jsonString := []byte{}
		if obj != nil {
			var err error
			jsonString, err = json.Marshal(obj)
			if err != nil {
				code := http.StatusInternalServerError
				return response(http.StatusText(code), code), nil
			}
		}

		return response(string(jsonString), code), nil
	}
}

func response(body string, code int) events.APIGatewayProxyResponse {
	headers := map[string]string{"Access-Control-Allow-Origin": "*"}
	return events.APIGatewayProxyResponse{Body: body, StatusCode: code, Headers: headers}
}
