package lambda

import (
	"context"
	"encoding/json"
	"strings"

	"code.justin.tv/vodsvc/aws/batch"
	"code.justin.tv/vodsvc/vhs/src/vhslog"

	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
)

type BatchEventFunc func(context.Context, batch.BatchMessage)

func StartBatchEvent(f BatchEventFunc) {
	if inLambda() {
		lambda.Start(wrapBatchEvent(f))
	} else {
		batchEventLocal(f)
	}
}

func batchEventLocal(f BatchEventFunc) {
	params := parseArgs()
	fakeBatchMessage := batch.BatchMessage{
		Status:     strings.ToUpper(params["status"]),
		Parameters: params,
	}

	f(context.TODO(), fakeBatchMessage)
}

// A convenience layer that executes common code snippets for lambda functions
// that implement AWS Batch Events (triggered through CloudWatch)
func wrapBatchEvent(f BatchEventFunc) CloudWatchEventFunc {
	return func(c context.Context, e events.CloudWatchEvent) {
		msg := batch.BatchMessage{}
		err := json.Unmarshal(e.Detail, &msg)
		if err != nil {
			vhslog.Logger.Fatal("Error unmarshalling job detail to JSON: " + err.Error())
		}

		f(c, msg)
	}
}
