package lambda

import (
	"errors"
	"fmt"
	"regexp"

	"github.com/aws/aws-lambda-go/events"
)

// GetAccountID retrieves the AWS Account ID from the context of a Lambda request
func GetAccountID(r events.APIGatewayProxyRequest) (string, error) {
	// TODO: Extract auth into shared fn or middleware after we define in spec
	accountID := r.RequestContext.Identity.AccountID
	if accountID == "" {
		return accountID, errors.New("getAccountId: unable to retrieve IAM account ID")
	}
	logger.Info(fmt.Sprintf("Invoked by %s on account %s", r.RequestContext.Identity.UserArn, accountID))
	return accountID, nil
}

// maxContentIDLength defines the maximum length of a content id
const maxContentIDLength = 40

// contentIDRegex defines the format of a content Identifier
var contentIDRegex = regexp.MustCompile(`([A-Za-z0-9\-\_]+)`)

// IsValidContentID validates a contentID only has letter, numbers, underscores or dashes
// and is smaller than 40 characters
func IsValidContentID(contentID string) bool {
	parsed := contentIDRegex.FindAllString(contentID, -1)

	if len(parsed) != 1 {
		return false
	}

	// ContentID Regex should only find a single match.
	if parsed[0] == contentID && (len(parsed[0]) < maxContentIDLength) {
		return true
	}

	return false
}
