package types

import (
	"time"
)

const (
	StatusCreated          = "created"
	StatusUploading        = "uploading"
	StatusPendingTranscode = "pendingTranscode"
	StatusTranscoding      = "transcoding"
	StatusFailed           = "failed"
	StatusProcessed        = "processed"
	StatusCancelled        = "cancelled"
)

const (
	TypeUpload = "upload"
)

const (
	ErrInternalFailure ErrCode = "internal_failure"
)

type ErrCode string

// Vod Types

type Upload struct {
	Id            string      `json:"id" dynamodbav:"Id"`
	S3Bucket      string      `json:"s3Bucket" dynamodbav:"S3Bucket"`
	S3Prefix      string      `json:"s3Prefix" dynamodbav:"S3Prefix"`
	VideoBucket   string      `json:"-" dynamodbav:"VideoBucket"`
	Size          int         `json:"size" dynamodbav:"Size"`
	Status        string      `json:"status" dynamodbav:"Status"`
	FailureReason string      `json:"failureReason,omitempty" dynamodbav:"FailureReason"`
	StatusTimes   StatusTimes `json:"statusTimes" dynamodbav:"StatusTimes"`
	LastUpdated   time.Time   `json:"-" dynamodbav:"LastUpdated"`
	SourceFile    string      `json:"-" dynamodbav:"SourceFile"`
}

type CreateUploadResponse struct {
	Upload      Upload      `json:"upload"`
	Credentials Credentials `json:"credentials"`
	S3URI       string      `json:"s3Uri"`
}

type ListUploadsResponse struct {
	Uploads         []Upload `json:"uploads"`
	PaginationToken string   `json:"paginationToken"`
	Count           int      `json:"count"`
}

type ListVideosResponse struct {
	Videos          []Video `json:"videos"`
	PaginationToken string  `json:"paginationToken"`
	Count           int     `json:"count"`
}

type Credentials struct {
	AccessKeyId     string `json:"accessKeyId"`
	SecretAccessKey string `json:"secretAccessKey"`
	SessionToken    string `json:"sessionToken"`
	Expiration      string `json:"expiration"`
}

type Video struct {
	CreateTime time.Time   `json:"createTime" dynamodbav:"CreateTime"`
	Duration   int         `json:"duration" dynamodbav:"Duration"`
	Id         string      `json:"id" dynamodbav:"Id"`
	S3Bucket   string      `json:"s3Bucket" dynamodbav:"S3Bucket"`
	S3Prefix   string      `json:"s3Prefix" dynamodbav:"S3Prefix"`
	VideoType  string      `json:"videoType" dynamodbav:"VideoType"`
	Renditions []Rendition `json:"renditions" dynamodbav:"Renditions"`
	Thumbnails []Thumbnail `json:"thumbnails" dynamodbav:"Thumbnails"`
	DeleteAt   *time.Time  `json:"deleteTime,omitempty" dynamodbav:"-"`
	DeleteDate string      `json:"-" dynamodbav:"DeleteDate,omitempty"`
	DeleteTime string      `json:"-" dynamodbav:"DeleteTime,omitempty"`
}

type Rendition struct {
	AudioCodec string `json:"audioCodec" dynamodbav:"AudioCodec"`
	VideoCodec string `json:"videoCodec" dynamodbav:"VideoCodec"`
	Bitrate    int32  `json:"bitrate" dynamodbav:"Bitrate"`
	Height     int32  `json:"height" dynamodbav:"Height"`
	Width      int32  `json:"width" dynamodbav:"Width"`
	FPS        string `json:"fps" dynamodbav:"FPS"`
	Label      string `json:"label" dynamodbav:"Label"`
	Manifest   string `json:"manifest" dynamodbav:"Manifest"`
}

type Thumbnail struct {
	Path   string  `json:"path" dynamodbav:"Path"`
	Offset float64 `json:"offset" dynamodbav:"Offset"`
}

type StatusTimes map[string]time.Time

// Live Types

type GetStreamResponse struct {
	Stream Stream `json:"stream"`
}

type Stream struct {
	// Custom identifier for an individual stream(broadcast)
	ContentID string `json:"contentID"`
	// Custom identifier for an individual stream(broadcast)
	CustomerID string `json:"customerID"`
	// Url to use to playback video for this stream
	PlaybackURL string `json:"playbackURL"`
	// URL to use to fetch the last generated thumbnail for this stream
	ThumbnailURL string `json:"thumbnailURL"`
	// RFC3339 timestamp noting when the stream started
	StartTime time.Time `json:"startTime"`
	// Same value as now()-stream_start_time truncated to second accuracy
	DurationSeconds int64 `json:"durationSeconds"`
	// The current operational state of this stream
	Status string `json:"status"`
	// The current ingestion health of this stream
	HealthStatus string `json:"healthStatus"`
	// The reason for the current health status
	HealthReason string `json:"healthReason"`
}

type ListStreamsResponse struct {
	Streams []Stream `json:"streams"`
}

type CreateStreamKeyRequest struct {
	ContentID               string `json:"contentID"`
	S3Bucket                string `json:"s3Bucket"`
	S3Prefix                string `json:"s3Prefix"`
	SnsNotificationEndpoint string `json:"snsNotificationEndpoint"`
	TTLSeconds              int    `json:"ttlSeconds"`
	CdnURL                  string `json:"cdnURL"`
}

type CreateStreamKeyResponse struct {
	ContentID         string                    `json:"contentID"`
	CustomerID        string                    `json:"customerID"`
	PlaybackURL       string                    `json:"playbackURL"`
	RtmpIngestURL     string                    `json:"rtmpIngestURL"`
	RtmpServer        string                    `json:"rtmpServer"`
	Streamkey         string                    `json:"streamkey"`
	StreamkeyMetadata StreamkeyMetadataResponse `json:"streamkeyMetadata"`
}

type StreamkeyMetadataResponse struct {
	ExpirationTime          time.Time `json:"expirationTime"`
	SnsNotificationEndpoint string    `json:"snsNotificationEndpoint"`
	TTLSeconds              int       `json:"ttlSeconds"`
	CdnURL                  string    `json:"cdnURL"`
}
