package vhsclient

import (
	"fmt"
	"io"
	"net/http"
	"strings"

	"code.justin.tv/vodsvc/vhs/src/types"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
)

func (c *Client) CreateUpload() (types.CreateUploadResponse, error) {
	path := "/uploads"
	upload := types.CreateUploadResponse{}

	err := c.requestJSON(http.MethodPost, path, &upload)
	if err != nil {
		return upload, fmt.Errorf("Error creating upload: " + err.Error())
	}

	c.logger.Log("CreateUpload; id=", upload.Upload.Id)
	return upload, nil
}

func (c *Client) UploadVideo(upload types.CreateUploadResponse, file io.Reader) error {
	creds := credentials.NewStaticCredentials(
		upload.Credentials.AccessKeyId,
		upload.Credentials.SecretAccessKey,
		upload.Credentials.SessionToken)
	config := aws.NewConfig().WithCredentials(creds)
	sess := session.Must(session.NewSession(config))
	input := &s3manager.UploadInput{
		Bucket: aws.String(upload.Upload.S3Bucket),
		Key:    aws.String(keyFromURI(upload.S3URI, upload.Upload.S3Bucket)),
		Body:   file,
	}
	uploader := s3manager.NewUploader(sess)
	_, err := uploader.Upload(input)
	if err != nil {
		return fmt.Errorf("Error uploading file: " + err.Error())
	}
	c.logger.Log("UploadVideo; id=", upload.Upload.Id)
	return nil
}

func (c *Client) GetUpload(id string) (types.Upload, error) {
	path := "/uploads/" + id
	upload := types.Upload{}

	err := c.requestJSON(http.MethodGet, path, &upload)
	if err != nil {
		return upload, fmt.Errorf("Error getting upload: id=" + id + ", err=" + err.Error())
	}

	c.logger.Log("GetUpload; id=", upload.Id, ", status=", upload.Status)
	return upload, nil
}

func (c *Client) ListUploads(paginationToken ...string) (types.ListUploadsResponse, error) {
	path := "/uploads"
	if len(paginationToken) > 0 {
		path = path + "?paginationToken=" + paginationToken[0]
	}
	uploads := types.ListUploadsResponse{}

	err := c.requestJSON(http.MethodGet, path, &uploads)
	if err != nil {
		return uploads, fmt.Errorf("Error listing uploads: " + err.Error())
	}

	c.logger.Log("ListUploads; count=", uploads.Count)
	return uploads, nil
}

func keyFromURI(uri, bucket string) string {
	return strings.TrimPrefix(uri, "s3://"+bucket+"/")
}
