package vhsclient

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"
)

type Client struct {
	BaseURL string
	logger  Logger
}

func New(url string) *Client {
	if url == "" {
		url = os.Getenv("VHS_ENDPOINT") + "/prod"
	}
	return &Client{
		BaseURL: url,
		logger:  NewDefaultLogger(),
	}
}

func (c *Client) requestJSON(method, path string, obj interface{}) error {
	bytes, err := c.request(method, path)
	if err != nil {
		return err
	}

	if err := json.Unmarshal(bytes, obj); err != nil {
		return fmt.Errorf("Error unmarshaling json: " + err.Error())
	}

	return nil
}

func (c *Client) request(method, path string) ([]byte, error) {
	url := c.BaseURL + path
	client := http.Client{}
	req, err := http.NewRequest(method, url, bytes.NewBuffer([]byte{}))
	if err != nil {
		return []byte{}, err
	}

	rsp, err := client.Do(req)
	if err != nil {
		return []byte{}, err
	}
	defer rsp.Body.Close()
	return ioutil.ReadAll(rsp.Body)
}

func (c *Client) SetLogger(f func(...interface{})) {
	c.logger = LoggerFunc(f)
}

type Logger interface {
	Log(...interface{})
}

type LoggerFunc func(...interface{})

func (f LoggerFunc) Log(args ...interface{}) {
	f(args...)
}

func NewDefaultLogger() Logger {
	return &defaultLogger{
		logger: log.New(os.Stdout, "", log.LstdFlags),
	}
}

type defaultLogger struct {
	logger *log.Logger
}

func (l defaultLogger) Log(args ...interface{}) {
	l.logger.Println(args...)
}
