package goauthorization

import (
	"fmt"
	"net/http"
	"testing"
	"time"

	. "github.com/smartystreets/goconvey/convey"
)

var (
	decRS256, _ = NewDecoder("RS256", "tanookiben", "cartman", readFile("testing/id_rsa.pub"))
	decHS512, _ = NewDecoder("HS512", "tanookiben", "cartman", []byte("glitch"))
	decES256, _ = NewDecoder("ES256", "tanookiben", "cartman", readFile("testing/id_ecc.pub"))
)

func TestNewDecoder(t *testing.T) {
	Convey("initializes RS256", t, func() {
		aud := "tanookiben"
		iss := "benspotatoes"
		res, err := NewDecoder(rs256, aud, iss, readFile("testing/id_rsa.pub"))
		So(err, ShouldBeNil)

		dec := res.(*decoderImpl)
		So(dec.aud, ShouldEqual, aud)
		So(dec.iss, ShouldEqual, iss)
		So(dec.alg.Name(), ShouldEqual, rs256)
		So(dec.hdr.Algorithm, ShouldEqual, rs256)
	})

	Convey("initializes HS512", t, func() {
		aud := "tanookiben"
		iss := "benspotatoes"
		res, err := NewDecoder(hs512, aud, iss, []byte("duDudu"))
		So(err, ShouldBeNil)

		dec := res.(*decoderImpl)
		So(dec.aud, ShouldEqual, aud)
		So(dec.iss, ShouldEqual, iss)
		So(dec.alg.Name(), ShouldEqual, hs512)
		So(dec.hdr.Algorithm, ShouldEqual, hs512)
	})
}

func TestDecode(t *testing.T) {
	Convey("returns an error for an empty token", t, func() {
		_, err := decRS256.Decode("")
		So(err, ShouldNotBeNil)
		_, err = decHS512.Decode("")
		So(err, ShouldNotBeNil)
	})

	Convey("parses a valid RS256 token", t, func() {
		str, err := getToken(rs256, validTokenParams).String()
		if err != nil {
			panic(err)
		}
		tkn, err := decRS256.Decode(str)
		So(err, ShouldBeNil)
		So(tkn.GetID(), ShouldNotBeBlank)
	})

	Convey("parses a valid HS512 token", t, func() {
		str, err := getToken(hs512, validTokenParams).String()
		if err != nil {
			panic(err)
		}
		tkn, err := decHS512.Decode(str)
		So(err, ShouldBeNil)
		So(tkn.GetID(), ShouldNotBeBlank)
	})
}

func buildRequest(token string) *http.Request {
	req, err := http.NewRequest("GET", "http://www.twitch.tv", nil)
	if err != nil {
		panic(err)
	}
	req.Header.Add("Twitch-Authorization", token)
	return req
}

func TestParseToken(t *testing.T) {
	Convey("returns err if no token provided", t, func() {
		req, err := http.NewRequest("GET", "http://www.twitch.tv", nil)
		if err != nil {
			panic(err)
		}
		_, err = decRS256.ParseToken(req)
		So(err, ShouldEqual, ErrNoAuthorizationToken)
		_, err = decHS512.ParseToken(req)
		So(err, ShouldEqual, ErrNoAuthorizationToken)
		_, err = decES256.ParseToken(req)
		So(err, ShouldEqual, ErrNoAuthorizationToken)
	})

	Convey("returns error if invalid token", t, func() {
		req := buildRequest("invalid.token.rip")
		_, err := decRS256.ParseToken(req)
		So(err, ShouldNotBeNil)
		_, err = decHS512.ParseToken(req)
		So(err, ShouldNotBeNil)
		_, err = decES256.ParseToken(req)
		So(err, ShouldNotBeNil)
	})

	Convey("parses RS256 token in header", t, func() {
		tkn := getToken(rs256, validTokenParams)
		str, err := tkn.String()
		if err != nil {
			panic(err)
		}
		req := buildRequest(str)
		chk, err := decRS256.ParseToken(req)
		So(err, ShouldBeNil)
		So(chk.GetID(), ShouldEqual, tkn.GetID())
		So(chk.GetSubject(), ShouldEqual, tkn.GetSubject())
		// Use ShouldHappenWithin to deal with loss of nanosecond precision
		So(chk.GetIssuedAt(), ShouldHappenWithin, time.Second, tkn.GetIssuedAt())
	})

	Convey("parses HS512 token in header", t, func() {
		tkn := getToken(hs512, validTokenParams)
		str, err := tkn.String()
		if err != nil {
			panic(err)
		}
		req := buildRequest(str)
		chk, err := decHS512.ParseToken(req)
		So(err, ShouldBeNil)
		So(chk.GetID(), ShouldEqual, tkn.GetID())
		So(chk.GetSubject(), ShouldEqual, tkn.GetSubject())
		// Use ShouldHappenWithin to deal with loss of nanosecond precision
		So(chk.GetIssuedAt(), ShouldHappenWithin, time.Second, tkn.GetIssuedAt())
	})

	Convey("parses ES256 token in header", t, func() {
		tkn := getToken(es256, validTokenParams)
		str, err := tkn.String()
		if err != nil {
			panic(err)
		}
		req := buildRequest(str)
		chk, err := decES256.ParseToken(req)
		So(err, ShouldBeNil)
		So(chk.GetID(), ShouldEqual, tkn.GetID())
		So(chk.GetSubject(), ShouldEqual, tkn.GetSubject())
		// Use ShouldHappenWithin to deal with loss of nanosecond precision
		So(chk.GetIssuedAt(), ShouldHappenWithin, time.Second, tkn.GetIssuedAt())
	})
}

func TestValidate(t *testing.T) {
	Convey("RS256 returns no error for a valid token", t, func() {
		tkn := getToken(rs256, validTokenParams)
		err := decRS256.Validate(tkn, sampleClaims)
		So(err, ShouldBeNil)
	})

	Convey("RS256 returns error for expired token", t, func() {
		tkn := getToken(rs256, expiredTokenParams)
		err := decRS256.Validate(tkn, sampleClaims)
		So(err.Error(), ShouldContainSubstring, ErrExpiredToken.Error())
	})

	Convey("RS256 returns error for not ready token", t, func() {
		tkn := getToken(rs256, notReadyTokenParams)
		err := decRS256.Validate(tkn, sampleClaims)
		So(err.Error(), ShouldContainSubstring, ErrInvalidNbf.Error())
	})

	Convey("RS256 returns error for invalid audience", t, func() {
		dec, err := NewDecoder(rs256, "glitch", "cartman", readFile("testing/id_rsa.pub"))
		if err != nil {
			panic(err)
		}
		tkn := getToken(rs256, validTokenParams)
		err = dec.Validate(tkn, sampleClaims)
		So(err.Error(), ShouldContainSubstring, ErrInvalidAudience.Error())
		So(err.Error(), ShouldContainSubstring, fmt.Sprintf("expected glitch"))
	})

	Convey("RS256 returns error for invalid issuer", t, func() {
		dec, err := NewDecoder(rs256, "tanookiben", "glitch", readFile("testing/id_rsa.pub"))
		if err != nil {
			panic(err)
		}
		tkn := getToken(rs256, validTokenParams)
		err = dec.Validate(tkn, sampleClaims)
		So(err.Error(), ShouldContainSubstring, ErrInvalidIssuer.Error())
		So(err.Error(), ShouldContainSubstring, fmt.Sprintf("expected glitch"))
	})

	Convey("RS256 returns error for missing capability", t, func() {
		tkn := getToken(rs256, validTokenParams)
		err := decRS256.Validate(tkn, CapabilityClaims{
			"fake": CapabilityClaim{
				"foobar": "bazbar",
			},
		})
		So(err.Error(), ShouldContainSubstring, ErrMissingCapability.Error())
	})

	Convey("RS256 returns error for invalid capability", t, func() {
		badClaims := CapabilityClaims{
			"capability": CapabilityClaim{
				"param": "otherArg",
			},
		}
		tkn := getToken(rs256, validTokenParams)
		err := decRS256.Validate(tkn, badClaims)
		So(err.Error(), ShouldContainSubstring, ErrInvalidCapability.Error())
	})
}
