package goauthorization

import (
	"io/ioutil"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

func readFile(path string) []byte {
	raw, err := ioutil.ReadFile(path)
	if err != nil {
		panic(err)
	}
	return raw
}

var (
	encRS256, _ = NewEncoder("RS256", "cartman", readFile("testing/id_rsa.pem"))
	encHS512, _ = NewEncoder("HS512", "cartman", []byte("glitch"))
	encES256, _ = NewEncoder("ES256", "cartman", readFile("testing/id_ecc.pem"))
)

func getToken(alg string, params TokenParams) *AuthorizationToken {
	switch alg {
	case rs256:
		return encRS256.Encode(params)
	case hs512:
		return encHS512.Encode(params)
	case es256:
		return encES256.Encode(params)
	}
	return nil
}

func TestNewEncoder(t *testing.T) {
	Convey("initializes RS256", t, func() {
		iss := "cartman"
		res, err := NewEncoder(rs256, iss, readFile("testing/id_rsa.pem"))
		So(err, ShouldBeNil)

		enc := res.(*encoderImpl)
		So(enc.iss, ShouldEqual, iss)
		So(enc.alg.Name(), ShouldEqual, rs256)
		So(enc.hdr.Algorithm, ShouldEqual, rs256)
	})

	Convey("initializes HS512", t, func() {
		iss := "cartman"
		res, err := NewEncoder(hs512, iss, []byte("glitch"))
		So(err, ShouldBeNil)

		enc := res.(*encoderImpl)
		So(enc.iss, ShouldEqual, iss)
		So(enc.alg.Name(), ShouldEqual, hs512)
		So(enc.hdr.Algorithm, ShouldEqual, hs512)
	})

	Convey("initializes ES256", t, func() {
		iss := "cartman"
		res, err := NewEncoder(es256, iss, readFile("testing/id_ecc.pem"))
		So(err, ShouldBeNil)

		enc := res.(*encoderImpl)
		So(enc.iss, ShouldEqual, iss)
		So(enc.alg.Name(), ShouldEqual, es256)
		So(enc.hdr.Algorithm, ShouldEqual, es256)
	})
}
