package goauthorization

import (
	"testing"
	"time"

	. "github.com/smartystreets/goconvey/convey"
)

const (
	clientID = "tardertoots"
)

var (
	rs256 = "RS256"
	hs512 = "HS512"
	es256 = "ES256"

	sampleClaims = CapabilityClaims{
		"capability": CapabilityClaim{
			"param": "arg",
		},
	}

	validTokenParams = TokenParams{
		Exp:      time.Now().Add(time.Hour),
		Nbf:      time.Now().Add(-1 * time.Hour),
		Aud:      []string{"tanookiben"},
		Sub:      "benspotatoes",
		Claims:   sampleClaims,
		Fpc:      true,
		ClientID: clientID,
	}

	expiredTokenParams = TokenParams{
		Exp:      time.Now().Add(-1 * time.Hour),
		Nbf:      time.Now().Add(-1 * time.Hour),
		Aud:      []string{"tanookiben"},
		Sub:      "benspotatoes",
		Claims:   sampleClaims,
		Fpc:      true,
		ClientID: clientID,
	}

	notReadyTokenParams = TokenParams{
		Exp:      time.Now().Add(time.Hour),
		Nbf:      time.Now().Add(time.Hour),
		Aud:      []string{"tanookiben"},
		Sub:      "benspotatoes",
		Claims:   sampleClaims,
		Fpc:      true,
		ClientID: clientID,
	}
)

func TestSerializeAndString(t *testing.T) {
	Convey("returns a string for RS256 token", t, func() {
		token := getToken(rs256, validTokenParams)
		res, err := token.String()
		So(err, ShouldBeNil)
		So(res, ShouldNotBeBlank)
	})

	Convey("returns a string for HS512 token", t, func() {
		token := getToken(hs512, validTokenParams)
		res, err := token.String()
		So(err, ShouldBeNil)
		So(res, ShouldNotBeBlank)
	})

	Convey("returns a string for ES256 token", t, func() {
		token := getToken(es256, validTokenParams)
		res, err := token.String()
		So(err, ShouldBeNil)
		So(res, ShouldNotBeBlank)
	})

	Convey("serialize should match string", t, func() {
		token := getToken(hs512, validTokenParams)
		ser, serErr := token.Serialize()
		str, strErr := token.String()
		So(serErr, ShouldBeNil)
		So(strErr, ShouldBeNil)
		So(string(ser), ShouldEqual, str)
	})
}

func TestGetters(t *testing.T) {
	Convey("GetID returns JwtID", t, func() {
		token := getToken(hs512, validTokenParams)
		res := token.GetID()
		So(token.GetID(), ShouldEqual, res)
	})

	Convey("GetSubject returns Sub", t, func() {
		token := getToken(hs512, validTokenParams)
		res := token.GetSubject()
		So(token.GetSubject(), ShouldEqual, res)
	})

	Convey("GetIssuedAt returns IssuedAt", t, func() {
		token := getToken(hs512, validTokenParams)
		res := token.GetIssuedAt()
		So(token.GetIssuedAt(), ShouldResemble, res)
	})

	Convey("GetTokenClaims returns Claims", t, func() {
		token := getToken(hs512, validTokenParams)
		res, err := token.GetTokenClaims("capability")
		So(err, ShouldBeNil)
		So(res, ShouldResemble, CapabilityClaim{"param": "arg"})
		_, err = token.GetTokenClaims("nonCapability")
		So(err, ShouldNotBeNil)
	})

	Convey("IsFirstPartyClient returns FirstPartyClient", t, func() {
		token := getToken(hs512, validTokenParams)
		res := token.IsFirstPartyClient()
		So(res, ShouldBeTrue)
	})

	Convey("GetClientID returns ClientID", t, func() {
		token := getToken(hs512, validTokenParams)
		So(token.GetClientID(), ShouldEqual, clientID)
	})
}
