package jwt

import (
	"crypto/rsa"
	"testing"
)

// According to the NIST, an RSA key of 3072 bits is equivalent in security to an ECC key of size 256 bits.
// These benchmarks test the speed in which a pre-encoded jwt can be signed with a private key of both RSA-3072 and ECC-256.
//
// http://csrc.nist.gov/publications/nistpubs/800-57/sp800-57_part1_rev3_general.pdf

var msg = []byte("eyJ0eXAiOiJKV1QiLCJhbGciOiJFUzM4NCJ9eyJmb28iOiJiYXIifQngAfKMbJUh0WWubSIYe5GMsA-aHNKwFbJk_wq3lq23aPp8H2anb1rRILIzVR0gUf4a8WzDtrzmiikuPWyCS6CN4-PwdgTk-5nehC7JXqlaBZU05p3toM3nWCwm_LXcld")

func BenchmarkRSASign2048(b *testing.B) {
	key, _ := rsa.GenerateKey(rnd, 2048)
	alg := RS256(key)

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		alg.Sign(msg)
	}
}

func BenchmarkRSASign3072(b *testing.B) {
	key, _ := rsa.GenerateKey(rnd, 3072)
	alg := RS256(key)

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		alg.Sign(msg)
	}
}

func BenchmarkECDSASign256(b *testing.B) {
	key := GetECDSAPrivateKey("_key/ec256-private.pem", "_key/ec256-public.pem")
	alg := ES256(key)

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		alg.Sign(msg)
	}
}
