package main

import (
	"bufio"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"os/exec"
	"strings"

	"github.com/Jeffail/gabs"
	gitconfig "github.com/tcnksm/go-gitconfig"
	"github.com/urfave/cli"
)

var (
	apiToken     = "9916c5c17be744039297352bbe6faa1e"
	coverageName = "coverage.json"
	coveragePath = ""
)

func init() {
	cmd := cli.Command{
		Name:        "get-coverage",
		Usage:       "Gathers coverage reports and uploads them to codecov.",
		Description: fmt.Sprintf("Recursively gathers coverage reports from cur working dir and uploads them to CodeCov.  Will generate a '%s' with aggregate data about the coverage.", coverageName),
		Action:      getCoverage,
		Flags: []cli.Flag{
			cli.StringFlag{
				Name:  "repo",
				Usage: "Git org name. If not defined, then grabbed from env.",
			},
			cli.StringFlag{
				Name:  "branch",
				Usage: "Git repo name. If not defined ,then grabbed from env.",
			},
			cli.StringFlag{
				Name:  "commit",
				Usage: "Git commit hash. If not defined, then grabbed from env.",
			},
			cli.StringFlag{
				Name:  "output",
				Usage: fmt.Sprintf("Path to generate '%s' in. Must be a dir. Defaults to working dir.", coverageName),
			},
			cli.BoolFlag{
				Name:  "verbose, v",
				Usage: "Display output.",
			},
		},
	}
	App.Commands = append(App.Commands, cmd)
}

func getCoverage(c *cli.Context) error {

	// Determine our verbosity.
	IsVerbose = c.Bool("verbose")

	// Figure out where we're going to put our coverage file.
	coveragePath = c.String("output")
	if coveragePath != "" {
		if !strings.HasSuffix(coveragePath, "/") {
			coveragePath += "/"
		}
		if _, err := os.Stat(coveragePath); os.IsNotExist(err) {
			return cli.NewExitError(err, 0)
		}
	}
	coveragePath = coveragePath + coverageName

	// Determine if there's a .git directory within our working path.
	hasGitDir := true
	if _, err := os.Stat(".git"); os.IsNotExist(err) {
		hasGitDir = false
	}

	// Attempt to get the repoName and repoOwner from the command-line args.
	repoName := ""
	repoOwner := ""
	repoStr := c.String("repo")
	if repoStr != "" {
		s := strings.Split(repoStr, "/")
		if len(s) != 2 {
			return cli.NewExitError(fmt.Sprintf("Invalid repo name: %s", repoStr), 0)
		}
		repoOwner = s[0]
		repoName = s[1]
	} else {
		// If there were no command-line args then check the environment variables.
		gitUrlStr := os.Getenv("GIT_URL")
		if gitUrlStr != "" {
			s := strings.Split(gitUrlStr, ":")
			if len(s) == 2 {
				s = strings.Split(s[1], "/")
				repoOwner = s[0]
				repoName = strings.TrimSuffix(s[1], ".git")
			}
		}
	}
	// If we couldn't get the repo org/name from the command-line arg or env, then check the .git directory.
	if repoName == "" || repoOwner == "" {
		if hasGitDir {
			remoteURL, err := gitconfig.OriginURL()
			if err != nil {
				return cli.NewExitError(err, 0)
			}
			urls := strings.Split(remoteURL, ":")
			if len(urls) < 2 {
				return errors.New("Couldn't parse git URL")
			}
			urls = strings.Split(urls[1], "/")
			if len(urls) < 2 {
				return errors.New("Couldn't parse git URL")
			}
			repoOwner = urls[0]
			repoName = strings.TrimSuffix(urls[1], ".git")
		}
		if repoName == "" || repoOwner == "" {
			return cli.NewExitError("Failed to get the repository owner/name", 0)
		}
	}

	// Attempt to get the branch name from the command-line args.
	repoBranch := c.String("branch")
	// If we couldn't then try the environment variable.
	if repoBranch == "" {
		repoBranch = os.Getenv("GIT_BRANCH")
	}
	// If we got no branch, then exit.
	if repoBranch == "" {
		if hasGitDir {
			cmd := exec.Command("git")
			cmd.Args = []string{"git", "rev-parse", "--abbrev-ref", "HEAD"}
			cmdOutput, err := cmd.Output()
			repoBranch = strings.TrimSpace(string(cmdOutput[:]))
			if err != nil {
				return cli.NewExitError(err, 0)
			}
		}
		if repoBranch == "" {
			return cli.NewExitError("Failed to get the branch", 0)
		}
	}

	// Try to get the commit.
	repoCommit := c.String("commit")
	// If we couldn't then try the environment variable.
	if repoCommit == "" {
		repoCommit = os.Getenv("GIT_COMMIT")
	}
	// If we got no commit, then exit.
	if repoCommit == "" {
		if hasGitDir {
			cmd := exec.Command("git")
			cmd.Args = []string{"git", "rev-parse", "HEAD"}
			cmdOutput, err := cmd.Output()
			repoCommit = strings.TrimSpace(string(cmdOutput[:]))
			if err != nil {
				return cli.NewExitError(err, 0)
			}
		}
		if repoCommit == "" {
			return cli.NewExitError("Failed to get commit id", 0)
		}
	}

	LogMsg("Twerking code coverage for...")
	LogMsg(fmt.Sprintf("  Git Repo:   %s/%s", repoOwner, repoName))
	LogMsg(fmt.Sprintf("  Git Branch: %s", repoBranch))
	LogMsg(fmt.Sprintf("  Git Commit: %s", repoCommit))

	err := runBashScript(repoOwner, repoName)
	if err != nil {
		return cli.NewExitError(err, 0)
	}

	err = generateCoverageFile(repoOwner, repoName, repoCommit, coveragePath)
	if err != nil {
		return cli.NewExitError(err, 0)
	}

	return nil
}

func GetScript() (string, error) {
	cmd := exec.Command("curl")
	cmd.Args = []string{"curl", "-s", "https://codecov.internal.justin.tv/bash"}
	outputBytes, err := cmd.Output()
	if err != nil {
		return "", err
	}
	scriptPath := "/tmp/bash_script"
	err = ioutil.WriteFile(scriptPath, outputBytes, 0777)
	if err != nil {
		return "", err
	}
	return scriptPath, nil
}

func runBashScript(repoOwner string, repoName string) error {
	script, scriptErr := GetScript()
	if scriptErr != nil {
		return scriptErr
	}
	LogMsg("Running CodeCov Script:")
	LogMsg("------------------------")
	cmd := exec.Command(script)
	cmd.Args = []string{script, "-t", "ghe", "-r", fmt.Sprintf("%s/%s", repoOwner, repoName)}
	cmdReader, err := cmd.StdoutPipe()
	if err != nil {
		return err
	}
	scanner := bufio.NewScanner(cmdReader)
	go func() {
		for scanner.Scan() {
			fmt.Printf("%s\n", scanner.Text())
		}
	}()
	err = cmd.Start()
	if err != nil {
		return err
	}
	err = cmd.Wait()
	if err != nil {
		return err
	}

	LogMsg("------------------------")

	// Remove the script.
	err = os.Remove(script)
	LogMsg(fmt.Sprintf("Removing: %s", script))
	return err

}

type CoverageData struct {
	coverage string
	branches float64
	methods  float64
	files    float64
	hits     float64
	messages float64
	misses   float64
	lines    float64
	partials float64
	sessions float64
}

func generateCoverageFile(repoOwner string, repoName string, repoCommit string, filePath string) error {
	LogMsg("Querying CodeCov API for Coverage Summary")
	// Form our api string.
	apiURL := fmt.Sprintf("https://codecov.internal.justin.tv/api/ghe/%s/%s/commit/%s?access_token=%s", repoOwner, repoName, repoCommit, apiToken)
	apiCmd := exec.Command("curl")
	apiCmd.Args = []string{"curl", "-X", "GET", apiURL}
	cmdBytes, err := apiCmd.Output()
	if err != nil {
		return cli.NewExitError(err, 0)
	}

	jsonParsed, err := gabs.ParseJSON(cmdBytes)
	if err != nil {
		return cli.NewExitError(err, 0)
	}

	coverageData := new(CoverageData)
	ok := false
	coverageData.coverage, ok = jsonParsed.Path("commit.totals.c").Data().(string)

	if !ok {
		fmt.Println(coverageData)
		return cli.NewExitError("Could not generate coverage data!  Couldn't parse reponse from server.", 0)
	}

	coverageData.branches, _ = jsonParsed.Path("commit.totals.b").Data().(float64)
	coverageData.methods, _ = jsonParsed.Path("commit.totals.d").Data().(float64)
	coverageData.files, _ = jsonParsed.Path("commit.totals.f").Data().(float64)
	coverageData.hits, _ = jsonParsed.Path("commit.totals.h").Data().(float64)
	coverageData.messages, _ = jsonParsed.Path("commit.totals.M").Data().(float64)
	coverageData.misses, _ = jsonParsed.Path("commit.totals.m").Data().(float64)
	coverageData.lines, _ = jsonParsed.Path("commit.totals.n").Data().(float64)
	coverageData.partials, _ = jsonParsed.Path("commit.totals.p").Data().(float64)
	coverageData.sessions, _ = jsonParsed.Path("commit.totals.s").Data().(float64)

	jsonData := gabs.New()
	jsonData.Set(coverageData.coverage, "coverage")
	jsonData.Set(coverageData.branches, "branches")
	jsonData.Set(coverageData.methods, "methods")
	jsonData.Set(coverageData.files, "files")
	jsonData.Set(coverageData.hits, "hits")
	jsonData.Set(coverageData.misses, "misses")
	jsonData.Set(coverageData.lines, "lines")
	jsonData.Set(coverageData.partials, "partials")

	jsonStr := jsonData.StringIndent("", "  ")
	fmt.Println("Coverage Summary:")
	fmt.Println(jsonStr)

	// Write our file.
	fo, err := os.Create(filePath)
	defer fo.Close()
	if err != nil {
		return cli.NewExitError(err, 0)
	}
	fmt.Fprintln(fo, jsonStr)
	fmt.Printf("Generated: %s\n", filePath)
	return nil
}
